/* KinokoCanvasFramedObject.cc */
/* Created by Enomoto Sanshiro on 11 July 2000. */
/* Last updated by Enomoto Sanshiro on 9 January 2002. */


#include <sstream>
#include <algorithm>
#include <cmath>
#include <cstring>
#include "KinokoCanvasDataScale.hh"
#include "KinokoCanvasColorScale.hh"
#include "KinokoCanvasDrawableObject.hh"
#include "KinokoCanvasFramedObject.hh"

using namespace std;


TKinokoCanvasFramedObject::TKinokoCanvasFramedObject(TKinokoCanvas* Canvas)
: TKinokoCanvasDrawableObject(Canvas)
{
    _ContentBackgroundColorIndex = -1;
}

TKinokoCanvasFramedObject::~TKinokoCanvasFramedObject()
{
}

void TKinokoCanvasFramedObject::Initialize(void)
{
    TKinokoCanvasDrawableObject::Initialize();

    _TopMargin = (int) (3.5 * _LabelCharacterHeight);
    _BottomMargin = _LabelCharacterHeight;
    _RightMargin = _LabelCharacterWidth;
    _LeftMargin = _LabelCharacterWidth;

    _FrameOffsetX = 0;
    _FrameOffsetY = 0;
    _FrameWidth = 0;
    _FrameHeight = 0;

    _TitleOffsetX = 0;
    _TitleOffsetY = 0;
    
    _IsFrameValid = false;
    _IsFrameFrozen = false;
}

void TKinokoCanvasFramedObject::SetMargin(float Top, float Bottom, float Left, float Right)
{
    if (Top >= 0) {
	_TopMargin = (int) (Top * _LabelCharacterHeight);
    }
    if (Bottom >= 0) {
	_BottomMargin = (int) (Bottom * _LabelCharacterHeight);
    }
    if (Left >= 0) {
	_LeftMargin = (int) (Left * _LabelCharacterWidth);
    }
    if (Right >= 0) {
	_RightMargin = (int) (Right * _LabelCharacterWidth);
    }

    SetPosition(_OffsetX, _OffsetY, _Width, _Height);
}

void TKinokoCanvasFramedObject::SetPosition(int OffsetX, int OffsetY, int Width, int Height)
{
    TKinokoCanvasDrawableObject::SetPosition(OffsetX, OffsetY, Width, Height);

    _FrameOffsetX = _OffsetX + _LeftMargin;
    _FrameOffsetY = _OffsetY + _TopMargin;
    _FrameWidth = _Width - (_LeftMargin + _RightMargin);
    _FrameHeight = _Height - (_TopMargin + _BottomMargin);

    _TitleOffsetX = _FrameOffsetX + _FrameWidth / 2;
    _TitleOffsetY = _OffsetY + _TopMargin / 2;
    
    _DrawableXMin = _FrameOffsetX;
    _DrawableXMax = _FrameOffsetX + _FrameWidth;
    _DrawableYMin = _FrameOffsetY;
    _DrawableYMax = _FrameOffsetY + _FrameHeight;

    _IsFrameValid = ((_FrameWidth > 0) && (_FrameHeight > 0));
}

void TKinokoCanvasFramedObject::SetContentBackgroundColor(const std::string& ColorName)
{
    _ContentBackgroundColorIndex = _ImageArea->AllocateColor(ColorName);
}

void TKinokoCanvasFramedObject::FreezeFrame(float XMin, float XMax, float YMin, float YMax)
{
    SetCoordinate(
	min(XMin, XMax), min(YMin, YMax), fabs(XMax-XMin), fabs(YMax-YMin)
    );
    
    _IsFrameFrozen = true;
}

void TKinokoCanvasFramedObject::ThawFrame(void)
{
    _IsFrameFrozen = false;
}

void TKinokoCanvasFramedObject::Clear(void)
{
    TKinokoCanvasDrawableObject::Clear();

    if (_ContentBackgroundColorIndex >= 0) {
	int OldColor = _ImageArea->SetColor(_ContentBackgroundColorIndex);
	_ImageArea->DrawRectFill(
	    _FrameOffsetX, _FrameOffsetY, 
	    _FrameOffsetX + _FrameWidth, _FrameOffsetY + _FrameHeight
	);
	_ImageArea->SetColor(OldColor);
    }
}

float TKinokoCanvasFramedObject::CanvasXOf(float x)
{
    return (x - _XMin) / _XWidth * _FrameWidth + _FrameOffsetX;
}

float TKinokoCanvasFramedObject::CanvasYOf(float y)
{
    return (y - _YMin) / _YWidth * _FrameHeight + _FrameOffsetY;
}

float TKinokoCanvasFramedObject::XOf(float CanvasX)
{
    return (CanvasX - _FrameOffsetX) / _FrameWidth * _XWidth + _XMin;
}

float TKinokoCanvasFramedObject::YOf(float CanvasY)
{
    return (CanvasY - _FrameOffsetY) / _FrameHeight * _YWidth + _YMin;
}

int TKinokoCanvasFramedObject::ProcessCommand(const string& Command, istream& InputStream)
{
    int Result = 0;

    if (Command == "frame") {
	Result = ProcessFrameCommand(InputStream);
	DrawFrame();
    }
    else if (Command == "drawframe") {
	Result = ProcessDrawFrameCommand(InputStream);
    }
    else if (Command == "setmargin") {
	Result = ProcessSetMarginCommand(InputStream);
    }
    else {
	Result = TKinokoCanvasDrawableObject::ProcessCommand(Command, InputStream);
    }

    return Result;
}

int TKinokoCanvasFramedObject::ProcessSetCommand(const string& Name, const string& Value)
{
    int Result = 1;

    if (Name == "title") {
	Result = SetTitle(Value);
    }
    else if (Name == "contentbackground") {
	SetContentBackgroundColor(Value);
	Result = 1;
    }
    else {
	Result = TKinokoCanvasDrawableObject::ProcessSetCommand(Name, Value);
    }

    return Result;
}

int TKinokoCanvasFramedObject::ProcessFrameCommand(istream& InputStream)
{
    int Result = 0;

    float XMin, XMax, YMin, YMax;
    if (InputStream >> XMin >> XMax >> YMin >> YMax) {
	if (! _IsFrameFrozen) {
	    Result = SetCoordinate(
		min(XMin, XMax), min(YMin, YMax), 
		fabs(XMax-XMin), fabs(YMax-YMin)
	    );
	}
    }

    return Result;
}

int TKinokoCanvasFramedObject::ProcessDrawFrameCommand(istream& InputStream)
{
    return DrawFrame();
}

int TKinokoCanvasFramedObject::ProcessSetMarginCommand(istream& InputStream)
{
    int Result = 0;

    float Top, Bottom, Left, Right;
    if (InputStream >> Top >> Bottom >> Left >> Right) {
	SetMargin(Top, Bottom, Left, Right);
	Result = 1;
    }

    return Result;
}

int TKinokoCanvasFramedObject::SetTitle(const string& Title)
{
    _Title = Title;
    return 1;
}

int TKinokoCanvasFramedObject::DrawFrame(void)
{
    if (_Title.empty()) {
	return 1;
    }

    int OldColorIndex = _ImageArea->SetColor(_ColorIndexList[Color_Foreground]);
    int OldFontIndex = _ImageArea->SetFont(_FontIndexList[Font_Title]);

    if (strncmp(_Title.c_str(), "math:", 5) == 0) {
	_Typesetter->Typeset(
	    _TitleOffsetX, _TitleOffsetY, 
	    _MathTextComposerList[Font_Title]->Compose(
		_Title.substr(5, string::npos)
	    ), 
	    "cc"
	);
    }
    else {
	_ImageArea->DrawText(_TitleOffsetX, _TitleOffsetY, _Title, "cc");
    }

    _ImageArea->SetColor(OldColorIndex);
    _ImageArea->SetFont(OldFontIndex);

    return 1;
}



TKinokoCanvasColorScaledObject::TKinokoCanvasColorScaledObject(TKinokoCanvas* Canvas, TKinokoCanvasColorScale* ColorScale)
: TKinokoCanvasFramedObject(Canvas)
{
    _ColorScale = ColorScale;
    _ZScale = 0;
    _IsZScaleFactoringEnabled = false;
}

TKinokoCanvasColorScaledObject::~TKinokoCanvasColorScaledObject()
{
    delete _ZScale;
    delete _ColorScale;
}

void TKinokoCanvasColorScaledObject::Initialize(void)
{
    _ZScaleBoxWidth = 8;
    _ZScaleWidth = _ImageArea->TextWidthOf("00000") + _ZScaleBoxWidth;
    _RightMargin += _ZScaleWidth;

    _IsZScaleLog = false;
    _IsZScaleLabelDisabled = false;

    SetZCoordinate(0, 1);

    _NumberOfZScaleLabels = 0;
    _ZScale = new TKinokoCanvasDataScale(0, 1, _NumberOfZScaleLabels);

    _ColorScale->Initialize();
}

int TKinokoCanvasColorScaledObject::SetZCoordinate(float ZMin, float ZWidth)
{
    _ZMinOrg = ZMin;
    _ZWidthOrg = ZWidth;

    if (_IsZScaleLog) {
	if (ZMin <= 0) {
	    float ZMax = ZMin + ZWidth;
	    float ZMaxForLog = (ZMax <= 0) ? 1.0 : ZMax;
	    float ZMinForLog = min(ZMaxForLog / 100.0, 0.5);
	    ZMin = ZMinForLog;
	    ZWidth = ZMaxForLog - ZMinForLog;
	}

	_LogZMin = log10(ZMin);
	_LogZMax = log10(ZMin + ZWidth);
	_LogZWidth = _LogZMax - _LogZMin;
    }

    _ZMin = ZMin;
    _ZWidth = ZWidth;
    _ZMax = ZMin + ZWidth;

    return 1;
}

int TKinokoCanvasColorScaledObject::SetZScaleLinear(void)
{
    if (_IsZScaleLog) {
	_IsZScaleLog = false;
	SetZCoordinate(_ZMinOrg, _ZWidthOrg);
	delete _ZScale;
	_ZScale = new TKinokoCanvasDataScale(
	    _ZMin, _ZMax, _NumberOfZScaleLabels
	);
    }

    return 1;
}

int TKinokoCanvasColorScaledObject::SetZScaleLog(void)
{
    if (! _IsZScaleLog) {
	_IsZScaleLog = true;
	SetZCoordinate(_ZMinOrg, _ZWidthOrg);
	delete _ZScale;
	_ZScale = new TKinokoCanvasLogDataScale(
	    _ZMin, _ZMax, _NumberOfZScaleLabels
	);
    }

    return 1;
}

int TKinokoCanvasColorScaledObject::EnableZScaleLabel(void)
{
    _IsZScaleLabelDisabled = false;
    return 1;
}

int TKinokoCanvasColorScaledObject::DisableZScaleLabel(void)
{
    _IsZScaleLabelDisabled = true;
    return 1;
}

int TKinokoCanvasColorScaledObject::ProcessFrameCommand(istream& InputStream)
{
    int Result = TKinokoCanvasFramedObject::ProcessFrameCommand(InputStream);
    if (Result == 0) {
	return Result;
    }

    Result = 0;

    float ZMin, ZMax;
    if (InputStream >> ZMin >> ZMax) {
	if (ZMax > ZMin) {
	    Result = SetZCoordinate(ZMin, ZMax - ZMin);
	}
    }

    return Result;
}

int TKinokoCanvasColorScaledObject::DrawFrame(void)
{
    TKinokoCanvasFramedObject::DrawFrame();
    if (! _IsFrameValid) {
	return 1;
    }

    int NumberOfXScaleLabels = min(
	12, _FrameWidth / (7 * _LabelCharacterWidth)
    );
    NumberOfXScaleLabels = max(3, NumberOfXScaleLabels);
    _NumberOfZScaleLabels = min(
	_FrameHeight / (2 * _LabelCharacterHeight),
	(int) (2 * ((float) _FrameHeight / _FrameWidth) * NumberOfXScaleLabels)
    );
    _NumberOfZScaleLabels = max(3, _NumberOfZScaleLabels);

    int x1 = _FrameOffsetX + _FrameWidth + _ZScaleWidth;
    int y1 = _FrameOffsetY + _FrameHeight;
    int x0 = x1 - _ZScaleBoxWidth;
    int y0 = _FrameOffsetY;
    int ScaleHeight = y1 - y0;

    int OldColorIndex = _ImageArea->SetColor(_ColorIndexList[Color_Foreground]);
    int OldLineWidth =  _ImageArea->SetLineWidth(1);
    int OldLineStyle =  _ImageArea->SetLineStyle();
    int OldFontIndex = _ImageArea->SetFont(_FontIndexList[Font_Label]);

    _ImageArea->DrawRect(x0, y0, x1, y1);

    int NumberOfColors = 256;
    float ColorStep = 1.0 / NumberOfColors;
    for (float z = 0; z < 1.0; z += ColorStep) {
	_ImageArea->SetColor(_ColorScale->ColorIndexOf(z));
	_ImageArea->DrawRectFill(
	    x0, y1 - (int) (ScaleHeight * (z + ColorStep)),
	    x1, y1 - (int) (ScaleHeight * z)
	);
    }

    _ImageArea->SetColor(_ColorIndexList[Color_Foreground]);

    if (! _ZTitle.empty()) {
	if (strncmp(_ZTitle.c_str(), "math:", 5) == 0) {
	    _Typesetter->SetTextOrientation(270);
	    _Typesetter->Typeset(
		x1 + 0.5 * _LabelCharacterHeight, y0,
		_MathTextComposerList[Font_Label]->Compose(
		    _ZTitle.substr(5, string::npos)
		), 
		"tl"
	    );
	    _Typesetter->SetTextOrientation(0);
	}
	else {
	    int OldTextOrientation = _ImageArea->SetTextOrientation(270);
	    _ImageArea->DrawText(
		x1 + 0.5 * _LabelCharacterHeight, y0, 
		_ZTitle, 
		"tl"
	    );
	    _ImageArea->SetTextOrientation(OldTextOrientation);
	}
    }

    if (! _IsZScaleLabelDisabled) {
	_ZScale->Rescale(_ZMin, _ZMax, _NumberOfZScaleLabels);

	// Z Scale Factoring //
	if (_IsZScaleFactoringEnabled) {
	    string FactorLabel = _ZScale->FactorizeScaleLabel();
	    if (! FactorLabel.empty()) {
		_Typesetter->Typeset(
		    x0, y0 - 3, 
		    _MathTextComposerList[Font_Label]->Compose(FactorLabel),
		    "bl"
		);
	    }
	}
	
	// Z Scale Labels //
	int NumberOfZScaleDivisions = _ZScale->NumberOfDivisions();
	for (int i = 0; i < NumberOfZScaleDivisions; i++) {
	    float zs = _ZScale->DivisionValueOf(i), y;
	    if (! _IsZScaleLog) {
		y = y1 - (ScaleHeight * (zs - _ZMin) / _ZWidth);
	    }
	    else {
		y = y1 - (ScaleHeight * (log10(zs) - _LogZMin) / _LogZWidth);
	    }
	    _ImageArea->DrawLine(x0 - 3, y, x0, y);
	    
	    string Label = _ZScale->DivisionLabelOf(i);
	    _Typesetter->Typeset(
		x0 - 3, y, 
		_MathTextComposerList[Font_Label]->Compose(Label),
		"cr"
	    );
	}
	
	// Z Scale Ticks //
	int NumberOfZScaleTicks = _ZScale->NumberOfTicks();
	for (int i = 0; i < NumberOfZScaleTicks; i++) {
	    float zs = _ZScale->TickValueOf(i);
	    float y;
	    if (! _IsZScaleLog) {
		y = y1 - ScaleHeight * (zs - _ZMin) / _ZWidth;
	    }
	    else {
		y = y1 - ScaleHeight * (log10(zs) - _LogZMin) / _LogZWidth;
	    }
	    _ImageArea->DrawLine(x0 - 2, y, x0, y);
	}
    }
    
    _ImageArea->SetColor(OldColorIndex);
    _ImageArea->SetLineWidth(OldLineWidth);
    _ImageArea->SetLineStyle(OldLineStyle);
    _ImageArea->SetFont(OldFontIndex);

    return 1;
}
