/* KinokoCanvasObject.cc */
/* Created by Enomoto Sanshiro on 10 July 2000. */
/* Last updated by Enomoto Sanshiro on 23 January 2008. */


#include <iostream>
#include <string>
#include "KinokoShellConfig.hh"
#include "KinokoCanvas.hh"
#include "KinokoCanvasImageArea.hh"
#include "KinokoCanvasTypesetter.hh"
#include "KinokoCanvasObject.hh"

using namespace std;


TKinokoCanvasObject::TKinokoCanvasObject(TKinokoCanvas* Canvas)
{
    _Canvas = Canvas;
    _ImageArea = 0;

    _OffsetX = 0;
    _OffsetY = 0;
    _Width = 0;
    _Height = 0;

    for (int i = 0; i < _NumberOfEnumFonts; i++) {
	_MathTextComposerList[i] = 0;
    }
    _Typesetter = 0;
}

TKinokoCanvasObject::~TKinokoCanvasObject()
{
    for (int i = 0; i < _NumberOfEnumFonts; i++) {
	delete _MathTextComposerList[i];
    }

    delete _Typesetter;
}

void TKinokoCanvasObject::Initialize(void)
{
    TKinokoShellObject::Initialize();
    _ImageArea = _Canvas->ImageArea();

    _ColorList[Color_Foreground] = "";  // empty for default
    _ColorList[Color_Background] = "";  // empty for default

    _ColorList[Color_Black] = "black";
    _ColorList[Color_White] = "white";

#if USE_PANGO
    _FontList[Font_Normal] = make_pair(string("helvetica"), 12);
    _FontList[Font_Title] = make_pair(string("helvetica"), 16);
    _FontList[Font_Label] = make_pair(string("helvetica"), 14);
#else
    _FontList[Font_Normal] = make_pair(string("helvetica"), 14);
    _FontList[Font_Title] = make_pair(string("helvetica"), 20);
    _FontList[Font_Label] = make_pair(string("helvetica"), 16);
#endif

    for (unsigned i = 0; i < _NumberOfEnumColors; i++) {
	if (! _ColorList[i].empty()) {
	    _ColorIndexList[i] = _ImageArea->AllocateColor(_ColorList[i]);
	}
	else {
	    if (i == Color_Foreground) {
		_ColorIndexList[i] = _ImageArea->SetColor(-1);
	    }
	    else if (i == Color_Background) {
		_ColorIndexList[i] = _ImageArea->SetBackgroundColor(-1);
	    }
	    else {
		_ColorIndexList[i] = -1;
	    }
	}
    }
    _ImageArea->SetColor(_ColorIndexList[Color_Foreground]);
    _ImageArea->SetBackgroundColor(_ColorIndexList[Color_Background]);

    _LineStyleIndexList[LineStyle_Solid] = _ImageArea->CreateLineStyle("solid");
    _LineStyleIndexList[LineStyle_Dot] = _ImageArea->CreateLineStyle("dot");
    _LineStyleIndexList[LineStyle_Dash] = _ImageArea->CreateLineStyle("dash");

    for (unsigned i = 0; i < _NumberOfEnumFonts; i++) {
	_FontIndexList[i] = _ImageArea->LoadFont(
	    _FontList[i].first, _FontList[i].second
	);
	_MathTextComposerList[i] = new TKinokoCanvasMathTextComposer(
	    _ImageArea
	);
	_MathTextComposerList[i]->SetFont(
	    _FontList[i].first, _FontList[i].second
	);
    }

    _Typesetter = new TKinokoCanvasTypesetter(_ImageArea);

    _ImageArea->SetFont(_FontIndexList[Font_Label]);
    _LabelCharacterHeight = _ImageArea->TextHeightOf("0");
    _LabelCharacterWidth = _ImageArea->TextWidthOf("0");

    _ImageArea->SetFont(_FontIndexList[Font_Normal]);
    _NormalFontLineHeight = _ImageArea->TextHeightOf("Ap");
}

int TKinokoCanvasObject::Construct(istream& InputStream)
{
    int Result = 1;

    float OffsetX, OffsetY, Width, Height;
    if (InputStream >> OffsetX >> OffsetY >> Width >> Height) {
	OffsetX = (OffsetX / 100) * _ImageArea->Width();
	OffsetY = (OffsetY / 100) * _ImageArea->Height();
	Width = (Width / 100) * _ImageArea->Width();
	Height = (Height / 100) * _ImageArea->Height();
	
	SetPosition((int) OffsetX, (int) OffsetY, (int) Width, (int) Height);
	_Canvas->AddCanvasObject(this);
    }
    else {
	Result = 0;
    }

    return Result;
}

void TKinokoCanvasObject::SetPosition(int OffsetX, int OffsetY, int Width, int Height)
{
    _OffsetX = OffsetX;
    _OffsetY = OffsetY;
    _Width = Width;
    _Height = Height;
}

bool TKinokoCanvasObject::Includes(int X, int Y)
{
    return (
	(X > _OffsetX) && (Y > _OffsetY) && 
	(X < _OffsetX + _Width) && (Y < _OffsetY + _Height)
    );
}

vector<string> TKinokoCanvasObject::ActionList(void)
{
    return vector<string>();
}
    
int TKinokoCanvasObject::ProcessAction(const string& ActionName)
{
    return 0;
}

int TKinokoCanvasObject::ProcessRangeSelect(int X0, int Y0, int X1, int Y1)
{
    return 0;
}

void TKinokoCanvasObject::Clear(void)
{
    int OldColor = _ImageArea->SetBackgroundColor(
	_ColorIndexList[Color_Background]
    );

    _ImageArea->ClearArea(
	_OffsetX, _OffsetY, _OffsetX + _Width, _OffsetY + _Height
    );

    _ImageArea->SetBackgroundColor(OldColor);
}
