/* KinokoControl.cc */
/* Created by Enomoto Sanshiro on 1 October 2001. */
/* Last updated by Enomoto Sanshiro on 27 March 2002. */


#include <iostream>
#include <fstream>
#include <sstream>
#include <cstring>
#include <cctype>
#include <string>
#include <vector>
#include <map>
#include <set>
#include "MushXml.hh"
#include "MushFileSystem.hh"
#include "MushMisc.hh"
#include "KinokoShell.hh"
#include "KinokoControlWidget.hh"
#include "KinokoControlPanel.hh"
#include "KinokoControlScript.hh"
#include "KinokoControl.hh"

using namespace std;
using namespace mush;


TKinokoControl::TKinokoControl(TKinokoShellConnector* ShellConnector)
: TKinokoShell(ShellConnector)
{
    _Script = 0;
    _ShellScript = 0;

    _CurrentState = "initial";
}

TKinokoControl::~TKinokoControl()
{
    for (unsigned i = 0; i < _WidgetList.size(); i++) {
	delete _WidgetList[i];
    }

    delete _Script;
    delete _ShellScript;
}

void TKinokoControl::AddWidget(TKinokoControlWidget* Widget, const string& EnabledStateList)
{
    string Name = Widget->Name();
    if (_WidgetTable.count(Name) == 0) {
	_WidgetTable[Widget->Name()] = Widget;
	_WidgetList.push_back(Widget);
    }

    if (! EnabledStateList.empty()) {
	string State;
	set<string> StateSet;
	istringstream StateListStream(EnabledStateList);
	while (StateListStream >> State) {
	    StateSet.insert(State);
	}

	_EnabledStateList[Widget] = StateSet;
    }
}

void TKinokoControl::AddInputWidget(TKinokoControlWidget* Widget)
{
    _InputWidgetList.push_back(Widget);
    _InputValueCache.push_back("");
}

void TKinokoControl::AddImageWidget(TKinokoControlWidget* Widget)
{
    _ImageWidgetList.push_back(Widget);
}

void TKinokoControl::LoadControlScript(istream& ScriptText, int LineNumberOffset)
{
    if (_Script == 0) {
	if (_ShellScript == 0) {
	    _ShellScript = new TKinokoShellScript(
		CreatePopupWindow(), CreateFileSelectDialog()
	    );
	}
	_Script = new TKinokoControlScript(this);
	_Script->Merge(_ShellScript);
    }

    _Script->SetLineNumberOffset(LineNumberOffset);
    try {
        _Script->Parse(ScriptText);
	_Script->Execute();
    }
    catch (TScriptException &e) {
	HandleException(
	    TKinokoShellException("ScriptException: " + e.Message())
	);
    }
}

void TKinokoControl::OnClickEvent(TKinokoControlWidget* Widget)
{
    string Action = Widget->ActionOnClick();
    if (Action.empty()) {
	vector<string> ArgumentList;
	Invoke(Widget->Name(), ArgumentList);
    }
    else if (isspace(Action[0])) {
	;
    }
    else {
	ExecuteAction(Action);
    }
}

void TKinokoControl::OnFocusEvent(TKinokoControlWidget* Widget)
{
    string Action = Widget->ActionOnFocus();
    if (! Action.empty()) {
	ExecuteAction(Action);
    }
}

void TKinokoControl::OnBlurEvent(TKinokoControlWidget* Widget)
{
    string Action = Widget->ActionOnBlur();
    if (! Action.empty()) {
	ExecuteAction(Action);
    }
}

void TKinokoControl::RedrawImageWidgets(void)
{
    for (unsigned i = 0; i < _ImageWidgetList.size(); i++) {
	_ImageWidgetList[i]->Redraw();
    }
}

int TKinokoControl::UpdateWidgetValues(void)
{
    ostringstream MessageStream;
    MessageStream << ".update control ";
    for (unsigned i = 0; i < _InputWidgetList.size(); i++) {
	string Name = _InputWidgetList[i]->Name();
	string Value = ReplaceEscape(_InputWidgetList[i]->Value());

	if (Value != _InputValueCache[i]) {
	    _InputValueCache[i] = Value;
	    MessageStream << Name << "=" << Value << "&";
	}
    }
    MessageStream << ";" << endl;

    string Message = MessageStream.str();

    _ShellConnector->SendMessage((char*) Message.c_str(), Message.size());

    return 1;
}

int TKinokoControl::Invoke(const string& MethodName, vector<string>& ArgumentList)
{
    if (_Script != 0) {
	try {
	    _Script->ExecuteInvocationTask(MethodName);
	}
	catch (TScriptException &e) {
	    HandleException(
		TKinokoShellException("ScriptException: " + e.Message())
	    );
	}
    }

    UpdateWidgetValues();

    ostringstream MessageStream;
    MessageStream << ".invoke " << MethodName << " ";
    for (unsigned i = 0; i < ArgumentList.size(); i++) {
	MessageStream << "[" << i << "]" << "=" << ArgumentList[i] << "&";
    }
    MessageStream << ";" << endl;

    string Message = MessageStream.str();

    _ShellConnector->SendMessage((char*) Message.c_str(), Message.size());

    return 1;
}

int TKinokoControl::ExecuteAction(const string& ActionName, vector<string>* ArgumentList)
{
    try {
	if (_Script == 0) {
	    throw TScriptException(
		"TKinokoControl::ExecuteAction()",
		"no such action: " + ActionName
	    );
	}

	if (ArgumentList == 0) {
	    _Script->Execute(ActionName);
	}
	else {
	    vector<TParaValue*> ParaArgumentList;
	    for (unsigned i = 0; i < ArgumentList->size(); i++) {
		ParaArgumentList.push_back(
		    new TParaValue((*ArgumentList)[i])
		);
	    }
	    try {
		_Script->Execute(ActionName, ParaArgumentList);
	    }
	    catch (TScriptException &e) {
		for (unsigned i = 0; i < ParaArgumentList.size(); i++) {
		    delete ParaArgumentList[i];
		}
		throw;
	    }
	    for (unsigned i = 0; i < ParaArgumentList.size(); i++) {
		delete ParaArgumentList[i];
	    }
	}
    }
    catch (TScriptException &e) {
	HandleException(
	    TKinokoShellException("ScriptException: " + e.Message())
        );
    }

    return 1;
}

void TKinokoControl::InitializeWidgets(void)
{
    for (unsigned i = 0; i < _ImageWidgetList.size(); i++) {
	_ImageWidgetList[i]->Initialize();
	_ImageWidgetList[i]->Redraw();
    }
}

int TKinokoControl::OnStartup(void)
{
    if (_Script == 0) {
	return 0;
    }

    try {
	vector<TParaValue*> ArgumentList;
	_Script->ExecuteEventTask("startup", ArgumentList);
    }
    catch (TScriptException &e) {
	HandleException(
	    TKinokoShellException("ScriptException: " + e.Message())
        );
    }

    return 1;
}

int TKinokoControl::OnShutdown(void)
{
    if (_Script == 0) {
	return 0;
    }

    try {
	vector<TParaValue*> ArgumentList;
	_Script->ExecuteEventTask("shutdown", ArgumentList);
    }
    catch (TScriptException &e) {
	HandleException(
	    TKinokoShellException("ScriptException: " + e.Message())
	);
    }

    return 1;
}

int TKinokoControl::OnEverySecond(void)
{
    if (_Script == 0) {
	return 0;
    }

    try {
	_Script->ExecuteScheduledTask(TMushDateTime::SecSinceEpoch());
    }
    catch (TScriptException &e) {
	HandleException(
	    TKinokoShellException("ScriptException: " + e.Message())
	);
    }

    return 1;
}

int TKinokoControl::ProcessSystemCommand(const string& Command, std::istream& InputStream)
{
    int Result = 0;
    if (Command == ".openControlPanel") {
	Result = ProcessOpenControlPanelCommand(InputStream);
    }
    else if (Command == ".clearWidgetValues") {
	Result = ProcessClearWidgetValuesCommand(InputStream);
    }
    else if (Command == ".saveWidgetValues") {
	Result = ProcessSaveWidgetValuesCommand(InputStream);
    }
    else if (Command == ".loadWidgetValues") {
	Result = ProcessLoadWidgetValuesCommand(InputStream);
    }
    else if (Command == ".setWidgetValue") {
	Result = ProcessSetWidgetValueCommand(InputStream);
    }
    else if (Command == ".setWidgetAttribute") {
	Result = ProcessSetWidgetAttributeCommand(InputStream);
    }
    else if (Command == ".loadControlScript") {
	Result = ProcessLoadControlScriptCommand(InputStream);
    }
    else if (Command == ".executeAction") {
	Result = ProcessExecuteActionCommand(InputStream);
    }
    else if (Command == ".changeState") {
	Result = ProcessChangeStateCommand(InputStream);
    }
    else if (Command == ".openPopup") {
	Result = ProcessOpenPopupCommand(InputStream, false);
    }
    else if (Command == ".openQueryPopup") {
	Result = ProcessOpenPopupCommand(InputStream, true);
    }
    else {
	Result = TKinokoShell::ProcessSystemCommand(Command, InputStream);
    }

    return Result;
}

int TKinokoControl::ProcessOpenControlPanelCommand(istream& InputStream)
{
    string ControlPanelScriptFileName;
    if (InputStream >> ControlPanelScriptFileName) {
	Construct(ControlPanelScriptFileName);
    }

    return 1;
}

int TKinokoControl::ProcessClearWidgetValuesCommand(istream& InputStream)
{
    ClearWidgetValues();

    return 1;
}

int TKinokoControl::ProcessSaveWidgetValuesCommand(istream& InputStream)
{
    string FileName;
    if (InputStream >> FileName) {
	SaveWidgetValues(FileName);
    }

    return 1;
}

int TKinokoControl::ProcessLoadWidgetValuesCommand(istream& InputStream)
{
    string FileName;
    if (InputStream >> FileName) {
	LoadWidgetValues(FileName);
    }

    return 1;
}

int TKinokoControl::ProcessSetWidgetValueCommand(istream& InputStream)
{
    string WidgetName;
    if (InputStream >> WidgetName >> ws) {
	string Value;
	getline(InputStream, Value, ';');
	SetWidgetValue(WidgetName, Value);
    }

    return 1;
}

int TKinokoControl::ProcessSetWidgetAttributeCommand(istream& InputStream)
{
    string WidgetName, AttributeName;
    if (InputStream >> WidgetName >> AttributeName >> ws) {
	string Value;
	getline(InputStream, Value, ';');
	SetWidgetValue(WidgetName, Value);
    }

    return 1;
}

int TKinokoControl::ProcessExecuteActionCommand(istream& InputStream)
{
    string ActionName;
    if (! (InputStream >> ActionName)) {
	return 0;
    }

    string Argument;
    vector<string> ArgumentList;
    while (getline(InputStream >> ws, Argument, ',')) {
	ArgumentList.push_back(Argument);
    }

    int Result;
    if (ArgumentList.empty()) {
	Result = ExecuteAction(ActionName);
    }
    else {
	Result = ExecuteAction(ActionName, &ArgumentList);
    }

    return Result;
}

int TKinokoControl::ProcessLoadControlScriptCommand(istream& InputStream)
{
    int Result = 0;

    string ControlScriptFileName;
    if (InputStream >> ControlScriptFileName) {
	ifstream ControlScriptText(ControlScriptFileName.c_str());
	if (ControlScriptText) {
	    LoadControlScript(ControlScriptText);
	    Result = 1;
	}
    }

    return Result;
}

int TKinokoControl::ProcessChangeStateCommand(istream& InputStream)
{
    string StateName;
    if (InputStream >> StateName) {
	ChangeState(StateName);
    }

    return 1;
}

int TKinokoControl::ProcessOpenPopupCommand(istream& InputStream, bool IsQuery)
{
    TKinokoShellPopupWindow* PopupWindow = CreatePopupWindow();

    string Type;
    if (getline(InputStream >> ws, Type, ',')) {
	PopupWindow->SetType(Type);
    }

    string ActionList;
    if (getline(InputStream >> ws, ActionList, ',')) {
	istringstream ActionListStream(ActionList);
	string Action;
	while (ActionListStream >> Action) {
	    PopupWindow->AddAction(Action);
	}
    }

    string Message;
    getline(InputStream >> ws, Message, ';');
    PopupWindow->SetMessage(Message.c_str());

    bool IsModal = IsQuery;
    string Selection = PopupWindow->Open(IsModal);

    if (IsQuery) {
	string Reply = Selection + ";\n";
	_ShellConnector->SendMessage((char*) Reply.c_str(), Reply.size());
    }

    delete PopupWindow;

    return 1;
}

int TKinokoControl::ChangeState(const string& StateName)
{
    map<TKinokoControlWidget*, set<std::string> >::iterator i;
    for (i = _EnabledStateList.begin(); i != _EnabledStateList.end(); i++) {
	TKinokoControlWidget* Widget = (*i).first;

	if ((*i).second.count(StateName) > 0) {
	    Widget->Enable();
	}
	else {
	    Widget->Disable();
	}
    }

    if (_Script != 0) {
	try {
	    _Script->ExecuteTransitionTask(_CurrentState, StateName);
	}
	catch (TScriptException &e) {
	    HandleException(
		TKinokoShellException("ScriptException: " + e.Message())
	    );
	}
    }

    _CurrentState = StateName;

    return 1;
}

int TKinokoControl::SetWidgetValue(const string& WidgetName, const string& Value)
{
    if (_WidgetTable.count(WidgetName) == 0) {
	return 0;
    }
    _WidgetTable[WidgetName]->SetValue(Value);

    return 1;
}

int TKinokoControl::SetWidgetAttribute(const string& WidgetName, const string& AttributeName, const string& Value)
{
    if (_WidgetTable.count(WidgetName) == 0) {
	return 0;
    }
    _WidgetTable[WidgetName]->SetAttribute(AttributeName, Value);

    return 1;
}

int TKinokoControl::ClearWidgetValues(void)
{
    for (unsigned i = 0; i < _InputWidgetList.size(); i++) {
	_InputWidgetList[i]->SetValue("");
    }

    return 1;
}

int TKinokoControl::SaveWidgetValues(const string& FileName)
{
    ofstream File(FileName.c_str());
    if (! File) {
	return 0;
    }

    File << "<?xml version=\"1.0\"?>" << endl;
    File << endl;
    File << "<kcml-widget-values>" << endl;

    for (unsigned i = 0; i < _WidgetList.size(); i++) {
	string Name = _WidgetList[i]->Name();
	string Type = _WidgetList[i]->Type();
	string Value, RawValue = _WidgetList[i]->Value();
	bool IsDisabled = _WidgetList[i]->IsDisabled();
	for (
	    string::iterator i = RawValue.begin(); 
	    i != RawValue.end(); i++
	){
	    switch (*i) {
	      case '&':
		Value += "&amp;";
		break;
	      case '<':
		Value += "&lt;";
		break;
	      case '>':
		Value += "&gt;";
		break;
	      case '"':
		Value += "&quot;";
		break;
	      case '\'':
		Value += "&apos;";
		break;
	      default:
		Value += *i;
		break;
	    }
	}

	File << "    <widget name=\"" << Name << "\"";
        File << " type=\"" << Type << "\"";
	if (IsDisabled) {
	    File << " status=\"disabled\"";
	}
	File << ">";
	File << Value;
	File << "</widget>";
	File << endl;
    }
    File << "</kcml-widget-values>" << endl;

    return 1;
}

int TKinokoControl::LoadWidgetValues(const string& FileName)
{
    class TDocumentHandler: public sax::DocumentHandler {
      public:
	TDocumentHandler(TKinokoControl* Control): _Control(Control) {}
	virtual ~TDocumentHandler() {}
      public:
	virtual void startElement(
	    const std::string& name, 
	    const sax::AttributeList& attribute_list
	){
	    if (name == "widget") {
		_Name = attribute_list.getValue("name");
		_Type = attribute_list.getValue("type");
		_Value = "";
	    }
	    else if (name == "operation") {
		string Type = attribute_list.getValue("type");
		string Target = attribute_list.getValue("target");
		TKinokoControlWidget* Widget = _Control->LookupWidget(Target);
		if (Widget) {
		    if (Type == "click") {
			_Control->OnClickEvent(Widget);
		    }
		}
	    }
	}
	virtual void endElement(const std::string& name) 
        {
	    if (name == "widget") {
		if (_Type != "Label") {
		    _Control->SetWidgetValue(_Name, _Value);
		}
	    }
	}
	virtual void characters(const std::string& text) 
        {
	    _Value += text;
	}
      protected:
	TKinokoControl* _Control;
	std::string _Name, _Type, _Value;
    };

    if (! TMushFileAttribute(FileName).IsReadable()) {
	return 0;
    }

    try {
	TDocumentHandler Handler(this);
	mush::sax::Parser Parser;
	Parser.setDocumentHandler(&Handler);
	Parser.parse(FileName);
    }
    catch (TSystemCallException &e) {
	cerr << "ERROR: " << e << endl;
	return 0;
    }

    return 1;
}

const std::string& TKinokoControl::CurrentState(void)
{
    return _CurrentState;
}

TKinokoControlWidget* TKinokoControl::LookupWidget(const std::string& WidgetName)
{
    return _WidgetTable[WidgetName];
}

string TKinokoControl::ReplaceEscape(const string& Value)
{
    string Result;
    for (unsigned i = 0; i < Value.size(); i++) {
	switch (Value[i]) {
	  case '%':
	    Result += "%p";
	    break;
	  case '&':
	    Result += "%a";
	    break;
          case '=':
	    Result += "%e";
	    break;
	  case ';':
	    Result += "%s";
	    break;
	  default:
	    Result += Value[i];
	}
    }

    return Result;
}
