/* KinokoControlViewletMessenger.cc */
/* Created by Enomoto Sanshiro on 12 August 2007. */
/* Last updated by Enomoto Sanshiro on 12 August 2007. */


#include <string>
#include <vector>
#include "KumaVector.hh"
#include "KiscScript.hh"
#include "KinokoControl.hh"
#include "KinokoControlScript.hh"
#include "KinokoControlViewletMessenger.hh"

using namespace std;
using namespace kuma;


TKinokoControlCanvasViewletMessenger::TKinokoControlCanvasViewletMessenger(TKinokoControlCanvasViewlet* CanvasViewlet)
: TKinokoControlWidgetMessenger(CanvasViewlet)
{
    _CanvasViewlet = CanvasViewlet;
}

TKinokoControlCanvasViewletMessenger::~TKinokoControlCanvasViewletMessenger()
{
}

int TKinokoControlCanvasViewletMessenger::MethodIdOf(const std::string& MethodName)
{
    if (MethodName == "clear") {
	return MethodId_Clear;
    }
    else if (MethodName == "drawLine") {
	return MethodId_DrawLine;
    }
    else if (MethodName == "drawCircle") {
	return MethodId_DrawCircle;
    }
    else if (MethodName == "drawText") {
	return MethodId_DrawText;
    }

    return TKinokoControlWidgetMessenger::MethodIdOf(MethodName);
}

int TKinokoControlCanvasViewletMessenger::InvokeMethod(int MethodId, std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    int Result = 0;

    switch (MethodId) {
      case MethodId_Clear:
	Result = Clear(ArgumentList, ReturnValue);
	break;
      case MethodId_DrawLine:
	Result = DrawLine(ArgumentList, ReturnValue);
	break;
      case MethodId_DrawCircle:
	Result = DrawCircle(ArgumentList, ReturnValue);
	break;
      case MethodId_DrawText:
	Result = DrawText(ArgumentList, ReturnValue);
	break;
      default:
	Result = TKinokoControlWidgetMessenger::InvokeMethod(
	    MethodId, ArgumentList, ReturnValue
	);
    }
    
    return Result;
}

int TKinokoControlCanvasViewletMessenger::Clear(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    _CanvasViewlet->Clear();
    return 1;
}

int TKinokoControlCanvasViewletMessenger::DrawLine(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() < 4) {
	throw TScriptException(
	    "drawLine(float x0, float y0, float x1, float y1)", 
	    "too few argument[s]"
	);
    }

    float X0 = ArgumentList[0]->AsDouble();
    float Y0 = ArgumentList[1]->AsDouble();
    float X1 = ArgumentList[2]->AsDouble();
    float Y1 = ArgumentList[3]->AsDouble();

    _CanvasViewlet->DrawLine(X0, Y0, X1, Y1);

    return 1;
}

int TKinokoControlCanvasViewletMessenger::DrawCircle(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() < 3) {
	throw TScriptException(
	    "drawCircle(float x, float y, float r)", "too few argument[s]"
	);
    }

    float X = ArgumentList[0]->AsDouble();
    float Y = ArgumentList[1]->AsDouble();
    float Radius = ArgumentList[2]->AsDouble();

    _CanvasViewlet->DrawCircle(X, Y, Radius);

    return 1;
}

int TKinokoControlCanvasViewletMessenger::DrawText(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() < 3) {
	throw TScriptException(
	    "drawText(float x, float y, string Text)", "too few argument[s]"
	);
    }

    float X = ArgumentList[0]->AsDouble();
    float Y = ArgumentList[1]->AsDouble();
    string Text = ArgumentList[2]->AsString();

    _CanvasViewlet->DrawText(X, Y, Text);

    return 1;
}



TKinokoControlPlotViewletMessenger::TKinokoControlPlotViewletMessenger(TKinokoControlPlotViewlet* PlotViewlet)
: TKinokoControlWidgetMessenger(PlotViewlet)
{
    _PlotViewlet = PlotViewlet;
}

TKinokoControlPlotViewletMessenger::~TKinokoControlPlotViewletMessenger()
{
}

int TKinokoControlPlotViewletMessenger::MethodIdOf(const std::string& MethodName)
{
    if (MethodName == "clear") {
	return MethodId_Clear;
    }
    else if (MethodName == "setRange") {
	return MethodId_SetRange;
    }
    else if (MethodName == "resetRange") {
	return MethodId_ResetRange;
    }
    else if (MethodName == "guessRange") {
	return MethodId_GuessRange;
    }
    else if (MethodName == "setColor") {
	return MethodId_SetColor;
    }
    else if (MethodName == "setLineWidth") {
	return MethodId_SetLineWidth;
    }
    else if (MethodName == "setMarker") {
	return MethodId_SetMarker;
    }
    else if (MethodName == "setYScaleLog") {
	return MethodId_SetYScaleLog;
    }
    else if (MethodName == "setTimeTick") {
	return MethodId_SetTimeTick;
    }
    else if (MethodName == "draw") {
	return MethodId_Draw;
    }

    return TKinokoControlWidgetMessenger::MethodIdOf(MethodName);
}

int TKinokoControlPlotViewletMessenger::InvokeMethod(int MethodId, std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    int Result = 0;

    switch (MethodId) {
      case MethodId_Clear:
	Result = Clear(ArgumentList, ReturnValue);
	break;
      case MethodId_SetRange:
	Result = SetRange(ArgumentList, ReturnValue);
	break;
      case MethodId_ResetRange:
	Result = ResetRange(ArgumentList, ReturnValue);
	break;
      case MethodId_GuessRange:
	Result = GuessRange(ArgumentList, ReturnValue);
	break;
      case MethodId_SetColor:
	Result = SetColor(ArgumentList, ReturnValue);
	break;
      case MethodId_SetLineWidth:
	Result = SetLineWidth(ArgumentList, ReturnValue);
	break;
      case MethodId_SetMarker:
	Result = SetMarker(ArgumentList, ReturnValue);
	break;
      case MethodId_SetYScaleLog:
	Result = SetYScaleLog(ArgumentList, ReturnValue);
	break;
      case MethodId_SetTimeTick:
	Result = SetTimeTick(ArgumentList, ReturnValue);
	break;
      case MethodId_Draw:
	Result = Draw(ArgumentList, ReturnValue);
	break;
      default:
	Result = TKinokoControlWidgetMessenger::InvokeMethod(
	    MethodId, ArgumentList, ReturnValue
	);
    }
    
    return Result;
}

int TKinokoControlPlotViewletMessenger::Clear(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    _PlotViewlet->Clear();
    return 1;
}

int TKinokoControlPlotViewletMessenger::SetRange(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() < 4) {
	throw TScriptException(
	    "Plot::setRange()", "too few argument[s]"
	);
    }

    float XMin = ArgumentList[0]->AsDouble();
    float XMax = ArgumentList[1]->AsDouble();
    float YMin = ArgumentList[2]->AsDouble();
    float YMax = ArgumentList[3]->AsDouble();

    _PlotViewlet->SetRange(XMin, XMax, YMin, YMax);

    return 1;
}

int TKinokoControlPlotViewletMessenger::ResetRange(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    _PlotViewlet->ResetRange();
    return 1;
}

int TKinokoControlPlotViewletMessenger::GuessRange(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() < 2) {
	throw TScriptException("Plot::guessRange()", "too few argument[s]");
    }
    if (! ArgumentList[0]->IsList() || ! ArgumentList[1]->IsList()) {
	throw TScriptException("Plot::guessRange()", "invalid argument[s]");
    }
    if (ArgumentList[0]->AsValueList().size() != ArgumentList[1]->AsValueList().size()) {
	throw TScriptException("Plot::guessRange()", "inconsisitent list length");
    }

    vector<TParaValue>& XValueList = ArgumentList[0]->AsValueList();
    vector<TParaValue>& YValueList = ArgumentList[1]->AsValueList();

    TKumaVector XList, YList;
    for (unsigned i = 0; i < XValueList.size(); i++) {
	XList.Add(XValueList[i].AsDouble());
	YList.Add(YValueList[i].AsDouble());
    }

    _PlotViewlet->GuessRange(XList, YList);

    return 1;
}

int TKinokoControlPlotViewletMessenger::SetColor(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() < 1) {
	throw TScriptException("setColor()", "too few argument[s]");
    }

    string ColorName = ArgumentList[0]->AsString();

    _PlotViewlet->SetColor(ColorName);

    return 1;
}

int TKinokoControlPlotViewletMessenger::SetLineWidth(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() < 1) {
	throw TScriptException("setLineWidth()", "too few argument[s]");
    }

    int LineWidth = ArgumentList[0]->AsLong();
    
    _PlotViewlet->SetLineWidth(LineWidth);

    return 1;
}

int TKinokoControlPlotViewletMessenger::SetMarker(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    string MarkerName;
    int MarkerSize = 0;

    if (ArgumentList.size() > 0) {
	MarkerName = ArgumentList[0]->AsString();
    }
    if (ArgumentList.size() > 1) {
	MarkerSize = ArgumentList[1]->AsLong();
    }

    _PlotViewlet->SetMarker(MarkerName, MarkerSize);

    return 1;
}

int TKinokoControlPlotViewletMessenger::SetYScaleLog(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    _PlotViewlet->SetYScaleLog();

    return 1;
}

int TKinokoControlPlotViewletMessenger::SetTimeTick(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    long Epoch = 0;  // zero for absolute time
    string Format;   // "%H:%M", "%m %y" etc, empty for auto
    string Unit;     // "sec", "day", etc, "-" for auto
    long Step = 0;   // zero for auto

    if (ArgumentList.size() >= 1) {
	Epoch = ArgumentList[0]->AsLong();
    }
    if (ArgumentList.size() >= 2) {
	Format = ArgumentList[1]->AsString();
    }
    if (ArgumentList.size() >= 3) {
	Unit = ArgumentList[2]->AsString();
    }
    if (ArgumentList.size() >= 4) {
	Step = ArgumentList[3]->AsLong();
    }

    _PlotViewlet->SetTimeTick(Epoch, Format, Unit, Step);

    return 1;
}

int TKinokoControlPlotViewletMessenger::Draw(std::vector<TParaValue*>& ArgumentList, TParaValue& ReturnValue) throw(TScriptException)
{
    if (ArgumentList.size() < 2) {
	throw TScriptException("Plot::draw()", "too few argument[s]");
    }
    if (! ArgumentList[0]->IsList() || ! ArgumentList[1]->IsList()) {
	throw TScriptException("Plot::draw()", "invalid argument[s]");
    }
    if (ArgumentList[0]->AsValueList().size() != ArgumentList[1]->AsValueList().size()) {
	throw TScriptException("Plot::draw()", "inconsisitent list length");
    }

    vector<TParaValue>& XValueList = ArgumentList[0]->AsValueList();
    vector<TParaValue>& YValueList = ArgumentList[1]->AsValueList();

    TKumaVector XList, YList;
    for (unsigned i = 0; i < XValueList.size(); i++) {
	XList.Add(XValueList[i].AsDouble());
	YList.Add(YValueList[i].AsDouble());
    }

    _PlotViewlet->Draw(XList, YList);

    return 1;
}
