/* KinokoControlVisualWidget.cc */
/* Created by Enomoto Sanshiro on 24 April 2006. */
/* Last updated by Enomoto Sanshiro 1 May 2007. */


#include <iostream>
#include <sstream>
#include <string>
#include <map>
#include <algorithm>
#include <cmath>
#include "KinokoControlWidget.hh"
#include "KinokoCanvasImageArea.hh"
#include "KinokoControlVisualWidget.hh"

using namespace std;


template<class T> class TWidgetCreator: public TKinokoControlVisualWidgetCreator {
  public:
    virtual TKinokoControlVisualWidget* CreateInstance(const std::string& Name, std::map<std::string, std::string>& OptionTable, TKinokoControlImagePort* ImagePort) {
	return new T(Name, OptionTable, ImagePort);
    }
};



TKinokoControlVisualWidgetTable* TKinokoControlVisualWidgetTable::_Instance = 0;

TKinokoControlVisualWidgetTable::TKinokoControlVisualWidgetTable(void)
{
}

TKinokoControlVisualWidgetTable::~TKinokoControlVisualWidgetTable()
{
}

TKinokoControlVisualWidgetTable* TKinokoControlVisualWidgetTable::GetInstance(void)
{
    if (_Instance == 0) {
	_Instance = new TKinokoControlVisualWidgetTable();
	_Instance->ConstructTable();
    }

    return _Instance;
}

void TKinokoControlVisualWidgetTable::DestroyInstance(void)
{
    _Instance->DestroyTable();

    delete _Instance;
    _Instance = 0;
}

void TKinokoControlVisualWidgetTable::AddWidgetCreator(const std::string& WidgetTypeName, TKinokoControlVisualWidgetCreator* Creator)
{
    _CreatorTable[WidgetTypeName] = Creator;
}

TKinokoControlVisualWidgetCreator* TKinokoControlVisualWidgetTable::FindWidgetCreator(const std::string& WidgetTypeName)
{
    return _CreatorTable[WidgetTypeName];
}

void TKinokoControlVisualWidgetTable::ConstructTable(void)
{
    AddWidgetCreator(
	"visualdisplay", 
	new TWidgetCreator<TKinokoControlVisualDisplayWidget>()
    );
    AddWidgetCreator(
	"visuallightbutton", 
	new TWidgetCreator<TKinokoControlVisualLightButtonWidget>()
    );
    AddWidgetCreator(
	"visuallight", 
	new TWidgetCreator<TKinokoControlVisualLightButtonWidget>()
    );
    AddWidgetCreator(
	"visualbutton", 
	new TWidgetCreator<TKinokoControlVisualLightButtonWidget>()
    );
    AddWidgetCreator(
	"visualindicator", 
	new TWidgetCreator<TKinokoControlVisualIndicatorWidget>()
    );
    AddWidgetCreator(
	"visualmeter", 
	new TWidgetCreator<TKinokoControlVisualMeterWidget>()
    );
    AddWidgetCreator(
	"visualguage", 
	new TWidgetCreator<TKinokoControlVisualGuageWidget>()
    );
    AddWidgetCreator(
	"visualslider", 
	new TWidgetCreator<TKinokoControlVisualSliderWidget>()
    );
}

void TKinokoControlVisualWidgetTable::DestroyTable(void)
{
    map<string, TKinokoControlVisualWidgetCreator*>::iterator i;
    for (i = _CreatorTable.begin(); i != _CreatorTable.end(); i++) {
	delete i->second;
    }
}



TKinokoControlVisualDisplayWidget::TKinokoControlVisualDisplayWidget(const std::string& Name, std::map<std::string, std::string>& OptionTable, TKinokoControlImagePort* ImagePort)
: TKinokoControlVisualWidget(Name, OptionTable, ImagePort)
{
    _Label = OptionTable["label"];

    _Foreground = OptionTable["foreground"];
    _Background = OptionTable["background"];
    _Alignment = OptionTable["align"];

    if (_Foreground.empty()) {
	_Foreground = OptionTable["fg"];
    }
    if (_Background.empty()) {
	_Background = OptionTable["bg"];
    }
    if (_Alignment.empty()) {
	_Alignment = "left";
    }

    if (_Foreground.empty()) {
	_Foreground = "black";
    }
    if (OptionTable["fontsize"].empty()) {
	_FontSize = 20;
    }
}

TKinokoControlVisualDisplayWidget::~TKinokoControlVisualDisplayWidget()
{
}

int TKinokoControlVisualDisplayWidget::DefaultWidth(void)
{
    return 200;
}

int TKinokoControlVisualDisplayWidget::DefaultHeight(void)
{
    return 30;
}

void TKinokoControlVisualDisplayWidget::Initialize(void)
{
    TKinokoControlVisualWidget::Initialize();

    if (! _Label.empty()) {
	_Value = _Label;
    }

    _ImageArea->SetColor(
	_ImageArea->AllocateColor(DecodeColorValue(_Foreground))
    );
    if (! _Background.empty()) {
	_ImageArea->SetBackgroundColor(_ImageArea->AllocateColor(_Background));
    }

    SetValue(_Value);
}

void TKinokoControlVisualDisplayWidget::SetValue(const std::string& Value)
{
    _Value = Value;

    _ImageArea->Clear();

    if (_Alignment == "center") {
	_ImageArea->DrawText(_Width/2, _Height/2, _Value, "cc");
    }
    else if (_Alignment == "right") {
	_ImageArea->DrawText(_Width-10, _Height/2, _Value, "rc");
    }
    else {
	_ImageArea->DrawText(10, _Height/2, _Value, "lc");
    }

    _ImageArea->Redraw();
}

bool TKinokoControlVisualDisplayWidget::SetAttribute(const std::string& Name, const std::string& Value)
{
    if ((Name == "foreground") || (Name == "fg")) {
	_Foreground = Value;
	_ImageArea->SetColor(
	    _ImageArea->AllocateColor(DecodeColorValue(_Foreground))
	);
    }
    else if ((Name == "background") || (Name == "bg")) {
	_Background = Value;
	_ImageArea->SetBackgroundColor(_ImageArea->AllocateColor(_Background));
    }
    else {
	return TKinokoControlWidget::SetAttribute(Name, Value);
    }

    return true;
}

bool TKinokoControlVisualDisplayWidget::GetAttribute(const std::string& Name, std::string& Value)
{
    if ((Name == "foreground") || (Name == "fg")) {
	Value = _Foreground;
    }
    else if ((Name == "background") || (Name == "bg")) {
	Value = _Background;
    }
    else {
	return TKinokoControlWidget::GetAttribute(Name, Value);
    }

    return true;
}


TKinokoControlVisualLightButtonWidget::TKinokoControlVisualLightButtonWidget(const std::string& Name, std::map<std::string, std::string>& OptionTable, TKinokoControlImagePort* ImagePort)
: TKinokoControlVisualWidget(Name, OptionTable, ImagePort)
{
    _Label = OptionTable["label"];

    if (! (istringstream(OptionTable["size"]) >> _Radius)) {
	if (_Label.empty()) {
	    _Radius = 5;
	}
	else {
	    _Radius = 20;
	}
    }
}

TKinokoControlVisualLightButtonWidget::~TKinokoControlVisualLightButtonWidget()
{
}

int TKinokoControlVisualLightButtonWidget::DefaultWidth(void)
{
    if (_Label.empty()) {
	return (int) (2*_Radius+3);
    }
    else {
	return max((int) (2*_Radius+3), 50);
    }
}

int TKinokoControlVisualLightButtonWidget::DefaultHeight(void)
{
    if (_Label.empty()) {
	return (int) (2*_Radius+3);
    }
    else {
	return (int) (2*_Radius+25);
    }
}

void TKinokoControlVisualLightButtonWidget::Initialize(void)
{
    TKinokoControlVisualWidget::Initialize();

    _TextColorIndex = _ImageArea->AllocateColor("black");

    _IsBeingPressed = false;
    SetValue(_Value);
}

void TKinokoControlVisualLightButtonWidget::SetValue(const std::string& Value)
{
    _Value = Value;

    _CurrentColorIndex = _ImageArea->AllocateColor(DecodeColorValue(_Value));
    _IsBeingPressed ? DrawPressed() : DrawReleased();
}

void TKinokoControlVisualLightButtonWidget::OnButtonPress(int ButtonNumber)
{
    if (ButtonNumber == 1) {
	_IsBeingPressed = true;
	DrawPressed();
    }
}

void TKinokoControlVisualLightButtonWidget::OnButtonRelease(int ButtonNumber)
{
    if (ButtonNumber == 1) {
	_IsBeingPressed = false;
	DrawReleased();
    }
}

void TKinokoControlVisualLightButtonWidget::DrawReleased(void)
{
    _ImageArea->Clear();

    _ImageArea->SetColor(_WhiteIndex);
    _ImageArea->DrawCircleFill(_Width/2-2, _Radius, _Radius);
    _ImageArea->SetColor(_BlackIndex);
    _ImageArea->DrawCircleFill(_Width/2+2, _Radius+4, _Radius);

    _ImageArea->SetColor(_CurrentColorIndex);
    _ImageArea->DrawCircleFill(_Width/2, _Radius+2, _Radius);

    if (! _Label.empty()) {
	_ImageArea->SetColor(_TextColorIndex);
	_ImageArea->DrawText(_Width/2+2, 2*_Radius+10, _Label, "ct");
    }

    _ImageArea->Redraw();
}

void TKinokoControlVisualLightButtonWidget::DrawPressed(void)
{
    _ImageArea->Clear();

    _ImageArea->SetColor(_BlackIndex);
    _ImageArea->DrawCircleFill(_Width/2+2, _Radius+4, _Radius);

    _ImageArea->SetColor(_CurrentColorIndex);
    _ImageArea->DrawCircleFill(_Width/2+4, _Radius+6, _Radius);

    if (! _Label.empty()) {
	_ImageArea->SetColor(_TextColorIndex);
	_ImageArea->DrawText(_Width/2+2, 2*_Radius+10, _Label, "ct");
    }

    _ImageArea->Redraw();
}



TKinokoControlVisualIndicatorWidget::TKinokoControlVisualIndicatorWidget(const std::string& Name, std::map<std::string, std::string>& OptionTable, TKinokoControlImagePort* ImagePort)
: TKinokoControlVisualWidget(Name, OptionTable, ImagePort)
{
    _Label = OptionTable["label"];

    if (OptionTable["font"].empty()) {
	_FontName += "-bold";
    }
    if (OptionTable["fontsize"].empty()) {
	_FontSize = 20;
    }
}

TKinokoControlVisualIndicatorWidget::~TKinokoControlVisualIndicatorWidget()
{
}

int TKinokoControlVisualIndicatorWidget::DefaultWidth(void)
{
    return 80;
}

int TKinokoControlVisualIndicatorWidget::DefaultHeight(void)
{
    return 64;
}

void TKinokoControlVisualIndicatorWidget::Initialize(void)
{
    TKinokoControlVisualWidget::Initialize();

    _ImageArea->SetTextAdjustment("cc");
    _TextColorIndex = _ImageArea->AllocateColor("black");

    _IsBeingPressed = false;
    SetValue(_Value);
}

void TKinokoControlVisualIndicatorWidget::SetValue(const std::string& Value)
{
    _Value = Value;

    _CurrentColorIndex = _ImageArea->AllocateColor(DecodeColorValue(_Value));
    _IsBeingPressed ? DrawPressed() : DrawReleased();
}

void TKinokoControlVisualIndicatorWidget::OnButtonPress(int ButtonNumber)
{
    if (ButtonNumber == 1) {
	_IsBeingPressed = true;
	DrawPressed();
    }
}

void TKinokoControlVisualIndicatorWidget::OnButtonRelease(int ButtonNumber)
{
    if (ButtonNumber == 1) {
	_IsBeingPressed = false;
	DrawReleased();
    }
}

void TKinokoControlVisualIndicatorWidget::DrawReleased(void)
{
    _ImageArea->SetColor(_CurrentColorIndex);
    _ImageArea->DrawRectFill(0, 0, _Width, _Height);

    _ImageArea->SetLineWidth(4);
    _ImageArea->DrawLine(0, 0, _Width, 0);
    _ImageArea->DrawLine(0, 0, 0, _Height);

    _ImageArea->SetColor(_BlackIndex);
    _ImageArea->DrawLine(_Width, 0, _Width, _Height);
    _ImageArea->DrawLine(0, _Height, _Width, _Height);

    if (! _Label.empty()) {
	_ImageArea->SetColor(_TextColorIndex);
	_ImageArea->DrawText(_Width/2, _Height/2, _Label);
    }

    _ImageArea->Redraw();
}

void TKinokoControlVisualIndicatorWidget::DrawPressed(void)
{
    _ImageArea->Clear();

    _ImageArea->SetColor(_CurrentColorIndex);
    _ImageArea->DrawRectFill(2, 2, _Width+2, _Height+2);

    _ImageArea->SetColor(_BlackIndex);
    _ImageArea->SetLineWidth(2);
    _ImageArea->DrawLine(2, 2, _Width+2, 2);
    _ImageArea->DrawLine(2, 2, 2, _Height+2);

    if (! _Label.empty()) {
	_ImageArea->SetColor(_TextColorIndex);
	_ImageArea->DrawText(_Width/2+2, _Height/2+2, _Label);
    }

    _ImageArea->Redraw();
}



TKinokoControlVisualMeterWidget::TKinokoControlVisualMeterWidget(const std::string& Name, std::map<std::string, std::string>& OptionTable, TKinokoControlImagePort* ImagePort)
: TKinokoControlVisualWidget(Name, OptionTable, ImagePort)
{
    _Label = OptionTable["label"];

    string Angle = OptionTable["angle"];
    string Min = OptionTable["min"];
    string Max = OptionTable["max"];

    if (! (istringstream(OptionTable["min"]) >> _Min)) {
	_Min = 0;
    }
    if (! (istringstream(OptionTable["max"]) >> _Max)) {
	_Max = 1;
    }
    if (_Max - _Min == 0) {
	_Max = _Min + 1;
    }
    else if (_Max < _Min) {
	swap(_Min, _Max);
    }
    if (! (istringstream(OptionTable["mark"]) >> _ScaleMark)) {
	_ScaleMark = (_Max - _Min) / 10.0;
    }

    if (! (istringstream(OptionTable["size"]) >> _Radius)) {
	_Radius = 100;
    }
    _Angle *= M_PI/180.0;

    if (! (istringstream(OptionTable["angle"]) >> _Angle)) {
	_Angle = 120;
    }
    _Angle *= M_PI/180.0;
    if (_Angle < 0) {
	_Angle = fabs(_Angle);
    }
}

TKinokoControlVisualMeterWidget::~TKinokoControlVisualMeterWidget()
{
}

int TKinokoControlVisualMeterWidget::DefaultWidth(void)
{
    if (_Angle < M_PI) {
	return (int) (_Radius * 2*sin(_Angle/2) + 10);
    }
    else {
	return (int) (2*_Radius + 10);
    }
}

int TKinokoControlVisualMeterWidget::DefaultHeight(void)
{
    if (_Angle < M_PI) {
	return (int) (_Radius + 10);
    }
    else {
	return (int) (2*_Radius + 10);
    }
}

bool TKinokoControlVisualMeterWidget::AddDecoration(const std::string& Name, std::map<std::string, std::string>& OptionTable)
{
    if (Name == "range") {
	string Color = OptionTable["color"];
	float Low, High;
	if (! (istringstream(OptionTable["low"]) >> Low)) {
	    Low = _Min;
	}
	Low = max(Low, _Min);
	if (! (istringstream(OptionTable["high"]) >> High)) {
	    High = _Max;
	}
	High = min(High, _Max);
	if (! Color.empty() && (Low < High)) {
	    _RangeList.push_back(
		make_pair(make_pair(Color, (int) 0), make_pair(Low, High))
	    );
	}
    }
    else {
	return false;
    }

    return true;
}

void TKinokoControlVisualMeterWidget::Initialize(void)
{
    TKinokoControlVisualWidget::Initialize();

    _LabelFontId = _ImageArea->LoadFont(_FontName, 14);
    _ScaleFontId = _ImageArea->LoadFont(_FontName, 8);

    _ImageArea->SetTextAdjustment("cc");
    _ImageArea->SetLineWidth(3);

    _ColorIndex = _ImageArea->AllocateColor("red");

    if (_Angle < M_PI) {
	_Radius = min(_Radius, _Height-10);
	_Radius = min((double) _Radius, (double) (_Width-10) / (2*sin(_Angle/2)));
	_X0 = _Width / 2;
	_Y0 = _Height - 5;
    }
    else {
	_Radius = min(_Radius, (_Height-10)/2);
	_Radius = min(_Radius, (_Width-10)/2);
	_X0 = _Width / 2;
	_Y0 = _Height / 2;
    }

    for (unsigned i = 0; i < _RangeList.size(); i++) {
	int ColorIndex = _ImageArea->AllocateColor(_RangeList[i].first.first);
	_RangeList[i].first.second = ColorIndex;
    }

    SetValue(_Value);
}

void TKinokoControlVisualMeterWidget::SetValue(const std::string& Value)
{
    _Value = Value;

    _ImageArea->SetColor(_WhiteIndex);
    _ImageArea->DrawRectFill(0, 0, _Width, _Height);

    _ImageArea->SetColor(_BlackIndex);
    _ImageArea->SetLineWidth(4);
    _ImageArea->DrawLine(_Width, 0, _Width, _Height);
    _ImageArea->DrawLine(0, _Height, _Width, _Height);

    _ImageArea->SetLineWidth(1);
    int OldFontId = _ImageArea->SetFont(_ScaleFontId);

    float R = _Radius;

    float N = (_Max - _Min) / _ScaleMark;
    for (int i = 0; i <= N; i++) {
	float Theta = _Angle * (0.5-i/N);
	float Value = _Min + i * _ScaleMark;
	_ImageArea->DrawLine(
	    _X0 - R*sin(Theta), _Y0 - R*cos(Theta),
	    _X0 - (R-5)*sin(Theta), _Y0 - (R-5)*cos(Theta)
	);
	_ImageArea->DrawNumberText(
	    _X0 - (R-12)*sin(Theta), _Y0 - (R-12)*cos(Theta), 
	    Value, "cc"
	);

	for (int j = 1; j < 5; j++) {
	    if (i+j/5.0 > N) {
		break;
	    }
	    float Theta2 = _Angle * (0.5-(i+j/5.0)/N);
	    _ImageArea->DrawLine(
		_X0 - R*sin(Theta2), _Y0 - R*cos(Theta2),
		_X0 - (R-2)*sin(Theta2), _Y0 - (R-2)*cos(Theta2)
	    );
	}
    }

    if (! _Label.empty()) {
	_ImageArea->SetFont(_LabelFontId);
	_ImageArea->DrawText(_Width/2, 0.7*_Height, _Label);
    }

    for (unsigned i = 0; i < _RangeList.size(); i++) {
	float Low = _RangeList[i].second.first;
	float High = _RangeList[i].second.second;

	float Theta0 = _Angle * (0.5 - (Low - _Min) / (_Max - _Min));
	float Theta1 = _Angle * (0.5 - (High - _Min) / (_Max - _Min));

	_ImageArea->SetColor(_RangeList[i].first.second);

	//... TODO: make _ImageArea->DrawArc() and use it ...//
	for (double Theta = Theta1; Theta <= Theta0; Theta += 1.0/(10*R)) {
	    _ImageArea->DrawLine(
		_X0 - (R+0)*sin(Theta), _Y0 - (R+0)*cos(Theta),
		_X0 - (R+3)*sin(Theta), _Y0 - (R+3)*cos(Theta)
	    );
	}
    }

    float X;
    if (! (istringstream(_Value) >> X)) {
	X = _Min;
    }
    float Theta = _Angle * (0.5 - (X - _Min) / (_Max - _Min));
    _ImageArea->SetLineWidth(2);
    _ImageArea->SetColor(_ColorIndex);
    _ImageArea->DrawLine(
	_X0, _Y0,
	_X0 - (R-3)*sin(Theta), _Y0 - (R-3)*cos(Theta)
    );

    _ImageArea->SetColor(_BlackIndex);
    _ImageArea->DrawCircleFill(_X0, _Y0, 5);

    _ImageArea->Redraw();
    _ImageArea->SetFont(OldFontId);
}



TKinokoControlVisualGuageWidget::TKinokoControlVisualGuageWidget(const std::string& Name, std::map<std::string, std::string>& OptionTable, TKinokoControlImagePort* ImagePort)
: TKinokoControlVisualWidget(Name, OptionTable, ImagePort)
{
    _Label = OptionTable["label"];
    _IsScaleDisabled = (OptionTable.count("noscale") > 0);

    _Foreground = OptionTable["foreground"];
    _Background = OptionTable["background"];

    if (_Foreground.empty()) {
	_Foreground = OptionTable["fg"];
    }
    if (_Background.empty()) {
	_Background = OptionTable["bg"];
    }

    if (! (istringstream(OptionTable["length"]) >> _Length)) {
	_Length = 200;
    }
    if (! (istringstream(OptionTable["wideness"]) >> _Wideness)) {
	_Wideness = 15;
    }
    if (! (istringstream(OptionTable["margin"]) >> _Margin)) {
	_Margin = -1;
    }

    if (! (istringstream(OptionTable["min"]) >> _Min)) {
	_Min = 0;
    }
    if (! (istringstream(OptionTable["max"]) >> _Max)) {
	_Max = 1;
    }
    if (_Max - _Min == 0) {
	_Max = _Min + 1;
    }
    else if (_Max < _Min) {
	swap(_Min, _Max);
    }
    if (! (istringstream(OptionTable["mark"]) >> _ScaleMark)) {
	_ScaleMark = (_Max - _Min) / 10.0;
    }
}

TKinokoControlVisualGuageWidget::~TKinokoControlVisualGuageWidget()
{
}

int TKinokoControlVisualGuageWidget::DefaultWidth(void)
{
    if (_Margin >= 0) {
	return (int) (_Wideness + _Margin);
    }

    if (_IsScaleDisabled && _Label.empty()) {
	return (int) _Wideness + 4;
    }
    else {
	return (int) _Wideness + 40;
    }
}

int TKinokoControlVisualGuageWidget::DefaultHeight(void)
{
    int DefaultHeight;

    if (_IsScaleDisabled) {
	DefaultHeight = (int) (_Length + _Wideness);
    }
    else {
	DefaultHeight = (int) (_Length + _Wideness + 20);
    }

    if (! _Label.empty()) {
	DefaultHeight += 20;
    }

    return DefaultHeight;
}

bool TKinokoControlVisualGuageWidget::AddDecoration(const std::string& Name, std::map<std::string, std::string>& OptionTable)
{
    if (Name == "range") {
	string Color = OptionTable["color"];
	float Low, High;
	if (! (istringstream(OptionTable["low"]) >> Low)) {
	    Low = _Min;
	}
	Low = max(Low, _Min);
	if (! (istringstream(OptionTable["high"]) >> High)) {
	    High = _Max;
	}
	High = min(High, _Max);
	if (! Color.empty() && (Low < High)) {
	    _RangeList.push_back(
		make_pair(make_pair(Color, (int) 0), make_pair(Low, High))
	    );
	}
    }
    else {
	return false;
    }

    return true;
}

void TKinokoControlVisualGuageWidget::Initialize(void)
{
    TKinokoControlVisualWidget::Initialize();

    if (_Foreground.empty()) {
	_Foreground = "lightblue";
    }
    if (_Background.empty()) {
	_Background = "white";
    }

    _ImageArea->SetTextAdjustment("cc");
    _ImageArea->SetLineWidth(3);

    _ForegroundIndex = _ImageArea->AllocateColor(_Foreground);
    _BackgroundIndex = _ImageArea->AllocateColor(_Background);

    _LabelFontId = _ImageArea->LoadFont(_FontName, 14);
    _ScaleFontId = _ImageArea->LoadFont(_FontName, 12);

    for (unsigned i = 0; i < _RangeList.size(); i++) {
	int ColorIndex = _ImageArea->AllocateColor(_RangeList[i].first.first);
	_RangeList[i].first.second = ColorIndex;
    }

    SetValue(_Value);
}

void TKinokoControlVisualGuageWidget::SetValue(const std::string& Value)
{
    _Value = Value;

    float h = _Length;
    float w = _Wideness - 4;
    float top = w/2, bottom = h + w/2;
    float center = w/2 + 4;
    float left = center - w/2, right = center + w/2;

    if (! _Label.empty()) {
	top += 20;
	bottom += 20;
    }

    float X;
    if (! (istringstream(_Value) >> X)) {
	X = _Min - 1;
    }

    int OldColor = _ImageArea->SetColor(_BackgroundIndex);
    _ImageArea->DrawCircleFill(center, top, w/2);
    _ImageArea->DrawCircleFill(center, bottom, w/2);
    _ImageArea->DrawRectFill(left, top, right, bottom);

    _ImageArea->SetColor(_ForegroundIndex);
    if (X > _Min) {
	float Z = (X - _Min) / (_Max - _Min);
	Z = min(Z, (float) 1.0);
	Z = max(Z, (float) 0.0);
	Z *= h;
	
	_ImageArea->DrawCircleFill(center, bottom, w/2);
	if (Z < w/2) {
	    _ImageArea->SetColor(_BackgroundIndex);
	    _ImageArea->DrawRectFill(left, bottom-w/2, right, bottom);
	    _ImageArea->SetColor(_ForegroundIndex);
	}
	_ImageArea->DrawRectFill(left, bottom-Z, right, bottom);
    }
    if (X > _Max) {
	_ImageArea->DrawCircleFill(center, top, w/2);
    }

    _ImageArea->SetColor(_BlackIndex);

    if (! _IsScaleDisabled) {
	int OldLineWidth = _ImageArea->SetLineWidth(1);
	int OldFontId = _ImageArea->SetFont(_ScaleFontId);

	float N = (_Max - _Min) / _ScaleMark;
	for (int i = 0; i <= N; i++) {
	    float Z = i/N * h;
	    float X = _Min + i * _ScaleMark;
	    _ImageArea->DrawLine(right, bottom-Z, right+5, bottom-Z);
	    _ImageArea->DrawNumberText(right + 7, bottom-Z, X, "cl");

	    for (int j = 1; j < 5; j++) {
		if (i+j/5.0 > N) {
		    break;
		}
		float Z2 = (i+j/5.0)/N * h;
		_ImageArea->DrawLine(right, bottom-Z2, right+2, bottom-Z2);
	    }
	}
	_ImageArea->SetLineWidth(OldLineWidth);
	_ImageArea->SetFont(OldFontId);
    }

    if (! _Label.empty()) {
	int OldFontId = _ImageArea->SetFont(_LabelFontId);
	_ImageArea->DrawText(left, top-w/2-8, _Label, "lb");
	_ImageArea->SetFont(OldFontId);
    }

    for (unsigned i = 0; i < _RangeList.size(); i++) {
	float Low = _RangeList[i].second.first;
	float High = _RangeList[i].second.second;

	float Z0 = (Low - _Min) / (_Max - _Min) * h;
	float Z1 = (High - _Min) / (_Max - _Min) * h;

	_ImageArea->SetColor(_RangeList[i].first.second);
	_ImageArea->DrawRectFill(left-4, bottom-Z1, left-2, bottom-Z0);
    }

    _ImageArea->SetColor(OldColor);

    _ImageArea->Redraw();
}



TKinokoControlVisualSliderWidget::TKinokoControlVisualSliderWidget(const std::string& Name, std::map<std::string, std::string>& OptionTable, TKinokoControlImagePort* ImagePort)
: TKinokoControlVisualWidget(Name, OptionTable, ImagePort)
{
    _Foreground = OptionTable["foreground"];
    _Label = OptionTable["label"];
    _Selection = OptionTable["selection"];
    _IsScaleDisabled = (OptionTable.count("noscale") > 0);

    if (_Foreground.empty()) {
	_Foreground = OptionTable["fg"];
    }

    istringstream is(_Selection);
    string SelectionItem;
    while (is >> SelectionItem) {
	_SelectionTable[SelectionItem] = _SelectionList.size();
	_SelectionList.push_back(SelectionItem);
    }

    if (! (istringstream(OptionTable["length"]) >> _Length)) {
	if (_SelectionList.empty()) {
	    _Length = 200;
	}
	else {
	    _Length = 20 * (_SelectionList.size() - 1);
	}
    }
    if (! (istringstream(OptionTable["wideness"]) >> _Wideness)) {
	_Wideness = 20;
    }
    if (! (istringstream(OptionTable["margin"]) >> _Margin)) {
	_Margin = -1;
    }

    if (! (istringstream(OptionTable["min"]) >> _Min)) {
	_Min = 0;
    }
    if (! (istringstream(OptionTable["max"]) >> _Max)) {
	_Max = 1;
    }
    if (_Max - _Min == 0) {
	_Max = _Min + 1;
    }
    else if (_Max < _Min) {
	swap(_Min, _Max);
    }
    if (! (istringstream(OptionTable["mark"]) >> _ScaleMark)) {
	_ScaleMark = (_Max - _Min) / 10.0;
    }
    if (! (istringstream(OptionTable["step"]) >> _Step)) {
	_Step = _ScaleMark;
    }
}

TKinokoControlVisualSliderWidget::~TKinokoControlVisualSliderWidget()
{
}

int TKinokoControlVisualSliderWidget::DefaultWidth(void)
{
    if (_Margin >= 0) {
	return (int) (_Wideness + _Margin);
    }

    if (_IsScaleDisabled && _Label.empty()) {
	return (int) _Wideness;
    }
    else {
	return (int) _Wideness + 40;
    }
}

int TKinokoControlVisualSliderWidget::DefaultHeight(void)
{
    int DefaultHeight;

    if (_IsScaleDisabled) {
	DefaultHeight = (int) (_Length + _Wideness);
    }
    else {
	DefaultHeight = (int) (_Length + _Wideness + 20);
    }

    if (! _Label.empty()) {
	DefaultHeight += 20;
    }

    return DefaultHeight;
}

void TKinokoControlVisualSliderWidget::Initialize(void)
{
    TKinokoControlVisualWidget::Initialize();

    if (_Foreground.empty()) {
	_Foreground = "rgb:808080";
    }
    _ForegroundIndex = _ImageArea->AllocateColor(_Foreground);

    _ImageArea->SetTextAdjustment("cc");
    _ImageArea->SetLineWidth(3);

    _LabelFontId = _ImageArea->LoadFont(_FontName, 15);
    _ScaleFontId = _ImageArea->LoadFont(_FontName, 12);

    _CurrentValue = _Min;
    _CurrentSelectionIndex = 0;

    SetValue(_Value);
}

void TKinokoControlVisualSliderWidget::SetValue(const std::string& Value)
{
    _Value = Value;

    float h = _Length;
    float w = _Wideness;
    float top = w/2, bottom = h + w/2;
    float center = w/2;
    float left = center - w/2, right = center + w/2;

    if (! _Label.empty()) {
	top += 20;
	bottom += 20;
    }

    float Z;
    if (_SelectionList.empty()) {
	float X;
	if (istringstream(_Value) >> X) {
	    _CurrentValue = X;
	}
	else {
	    X = _CurrentValue;
	}
	Z = (X - _Min) / (_Max - _Min);
	Z = min(Z, (float) 1.0);
	Z = max(Z, (float) 0.0);
    }
    else if (_SelectionList.size() > 1) {
	if (_SelectionTable.count(_Value) > 0) {
	    _CurrentSelectionIndex = _SelectionTable[_Value];
	}
	Z = (float) _CurrentSelectionIndex / (_SelectionList.size()-1);
    }
    Z *= h;
	
    int OldColor = _ImageArea->SetColor(_BlackIndex);
    _ImageArea->Clear();
    _ImageArea->SetColor(_BlackIndex);
    _ImageArea->DrawLine(center, top, center, bottom);

    if (! _IsScaleDisabled) {
	_ImageArea->SetColor(_BlackIndex);
	int OldLineWidth = _ImageArea->SetLineWidth(1);
	int OldFontId = _ImageArea->SetFont(_ScaleFontId);

	if (_SelectionList.empty()) {
	    float N = (_Max - _Min) / _ScaleMark;
	    for (int i = 0; i <= N; i++) {
		float Z = i/N * h;
		float X = _Min + i * _ScaleMark;
		_ImageArea->DrawLine(center+3, bottom-Z, right+5, bottom-Z);
		_ImageArea->DrawNumberText(right + 7, bottom-Z, X, "cl");
		
		float N2 = _ScaleMark / _Step;
		for (int j = 1; j < N2; j++) {
		    if (i+j/N2 > N) {
			break;
		    }
		    float Z2 = (i+j/N2)/N * h;
		    _ImageArea->DrawLine(
			center+3, bottom-Z2, right+2, bottom-Z2
		    );
		}
	    }
	}
	else if (_SelectionList.size() > 1) {
	    int N = _SelectionList.size();
	    for (int i = 0; i < N; i++) {
		float Z = (float) i/(N-1) * h;
		_ImageArea->DrawLine(center+3, bottom-Z, right+5, bottom-Z);
		_ImageArea->DrawText(
		    right + 7, bottom-Z, _SelectionList[i], "cl"
		);
	    }
	}

	_ImageArea->SetLineWidth(OldLineWidth);
	_ImageArea->SetFont(OldFontId);
    }

    _ImageArea->SetColor(_ForegroundIndex);
    _ImageArea->DrawRectFill(left, bottom-Z-5, right, bottom-Z+5);
    _ImageArea->SetColor(_BlackIndex);
    _ImageArea->DrawLine(left, bottom-Z, right, bottom-Z);

    _ImageArea->SetColor(OldColor);

    if (! _Label.empty()) {
	int OldFontId = _ImageArea->SetFont(_LabelFontId);
	_ImageArea->SetTextAdjustment("lb");
	_ImageArea->DrawText(left, top-w/2, _Label, "lb");
	_ImageArea->SetFont(OldFontId);
    }

    _ImageArea->Redraw();
}

void TKinokoControlVisualSliderWidget::OnButtonPress(int ButtonNumber)
{
    if (ButtonNumber == 1) {
	if (_SelectionList.empty()) {
	    if (_CurrentValue + _Step <= _Max) {
		ostringstream os;
		os << (_CurrentValue+_Step);
		SetValue(os.str());
	    }
	}
	else {
	    _CurrentSelectionIndex++;
	    if (_CurrentSelectionIndex >= (int) _SelectionList.size()) {
		_CurrentSelectionIndex = 0;
	    }
	    SetValue(_SelectionList[_CurrentSelectionIndex]);
	}
    }
    else if (ButtonNumber == 3) {
	if (_SelectionList.empty()) {
	    if (_CurrentValue - _Step >= _Min) {
		ostringstream os;
		os << (_CurrentValue-_Step);
		SetValue(os.str());
	    }
	}
	else {
	    _CurrentSelectionIndex--;
	    if (_CurrentSelectionIndex < 0) {
		_CurrentSelectionIndex = _SelectionList.size() - 1;
	    }
	    SetValue(_SelectionList[_CurrentSelectionIndex]);
	}
    }
}

