/* KinokoCanvasImageAreaGtk.cc */
/* Created by Enomoto Sanshiro on 9 July 2000. */
/* Last updated by Enomoto Sanshiro on 12 April 2003. */


#include <sstream>
#include <string>
#include <vector>
#include <map>
#include <cmath>
#include <algorithm>
#include <gtk/gtk.h>
#include <gdk_imlib.h>
#include "KinokoCanvasColorScale.hh"
#include "KinokoCanvasImageAreaGtk.hh"

using namespace std;


TKinokoCanvasImageAreaGtk::TKinokoCanvasImageAreaGtk(GtkWidget* DrawingArea)
: TKinokoCanvasImageArea(DrawingArea->allocation.width, DrawingArea->allocation.height) 
{
    _DrawingArea = DrawingArea;
    _Drawable = gdk_pixmap_new(_DrawingArea->window, _Width, _Height, -1);

    _BlackGC = gdk_gc_new(_DrawingArea->window);
    _WhiteGC = gdk_gc_new(_DrawingArea->window);
    gdk_gc_copy(_BlackGC, _DrawingArea->style->black_gc);
    gdk_gc_copy(_WhiteGC, _DrawingArea->style->white_gc);

    _DoubleBufferGC = gdk_gc_new(_DrawingArea->window);
    gdk_gc_copy(_DoubleBufferGC, _DrawingArea->style->black_gc);

    _PointListBuffer = 0;
    _PointListBufferSize = 0;

    gdk_imlib_init();
    gtk_widget_push_visual(gdk_imlib_get_visual());
    gtk_widget_push_colormap(gdk_imlib_get_colormap());

    GdkColor* ForegroundColor = gdk_color_copy(&_DrawingArea->style->black);
    _ColorList.push_back(ForegroundColor);
    _CurrentColorIndex = 0;
    GdkColor* BackgroundColor = gdk_color_copy(&_DrawingArea->style->white);
    _ColorList.push_back(BackgroundColor);
    _CurrentBackgroundColorIndex = 1;

    _CurrentLineWidth = 1;

    _CurrentLineStyleIndex = 0;
    _LineStyleList.push_back(pair<signed char*, int>(0, 0));
    _CurrentLineSolidness = GDK_LINE_SOLID;
    _CurrentLineOffset = 0;

    _CurrentDrawingFunction = DrawingFunction_Copy;

    _CurrentFontIndex = 0;
    _CurrentFont = _DrawingArea->style->font;
    _FontList.push_back(_CurrentFont);

    _CurrentTextAdjustment = "bl";
    _CurrentTextOrientation = 0;

    _PredefinedColorTable["white"] = AllocateColorRgb(1.0, 1.0, 1.0);
    _PredefinedColorTable["black"] = AllocateColorRgb(0, 0, 0);
    _PredefinedColorTable["gray"] = AllocateColorRgb(0.5, 0.5, 0.5);
    _PredefinedColorTable["red"] = AllocateColorRgb(1.0, 0, 0);
    _PredefinedColorTable["green"] = AllocateColorRgb(0, 0.6, 0);
    _PredefinedColorTable["blue"] = AllocateColorRgb(0, 0, 1.0);
    _PredefinedColorTable["yellow"] = AllocateColorRgb(0.9, 0.8, 0);
    _PredefinedColorTable["cyan"] = AllocateColorRgb(0, 0.8, 0.8);
    _PredefinedColorTable["magenta"] = AllocateColorRgb(0.55, 0, 0.55);
    _PredefinedColorTable["orange"] = AllocateColorRgb(1.0, 0.65, 0);
    _PredefinedColorTable["brown"] = AllocateColorRgb(0.54, 0.27, 0.07);
    _PredefinedColorTable["bluegreen"] = AllocateColorRgb(0.13, 0.70, 0.67);
    _PredefinedColorTable["greenyellow"] = AllocateColorRgb(0.68, 1.0, 0.18);
    _PredefinedColorTable["wheat"] = AllocateColorRgb(0.8, 0.67, 0.49);
    _PredefinedColorTable["lightred"] = AllocateColorRgb(1.0, 0.5, 0.5);
    _PredefinedColorTable["lightgreen"] = AllocateColorRgb(0.5, 1.0, 0.5);
    _PredefinedColorTable["lightblue"] = AllocateColorRgb(0.5, 0.5, 1.0);
    _PredefinedColorTable["lightyellow"] = AllocateColorRgb(1.0, 1.0, 0.3);
    _PredefinedColorTable["lightcyan"] = AllocateColorRgb(0.5, 1.0, 1.0);
    _PredefinedColorTable["lightmagenta"] = AllocateColorRgb(1.0, 0.5, 1.0);
    _PredefinedColorTable["lightgray"] = AllocateColorRgb(0.83, 0.83, 0.83);

    _PredefinedFontNameTable["times"] = "-*-times-medium-r-normal";
    _PredefinedFontNameTable["times-roman"] = "-*-times-medium-r-normal";
    _PredefinedFontNameTable["times-italic"] = "-*-times-medium-i-normal";
    _PredefinedFontNameTable["times-bold"] = "-*-times-bold-r-normal";
    _PredefinedFontNameTable["times-bolditalic"] = "-*-times-bold-i-normal";
    _PredefinedFontNameTable["helvetica"] = "-*-helvetica-medium-r-normal";
    _PredefinedFontNameTable["helvetica-oblique"] = "-*-helvetica-medium-o-normal";
    _PredefinedFontNameTable["helvetica-bold"] = "-*-helvetica-bold-r-normal";
    _PredefinedFontNameTable["helvetica-boldoblique"] = "-*-helvetica-bold-o-normal";
    _PredefinedFontNameTable["courier"] = "-*-courier-medium-r-normal";
    _PredefinedFontNameTable["courier-oblique"] = "-*-courier-medium-o-normal";
    _PredefinedFontNameTable["courier-bold"] = "-*-courier-bold-r-normal";
    _PredefinedFontNameTable["courier-boldoblique"] = "-*-courier-bold-o-normal";
    _PredefinedFontNameTable["symbol"] = "-*-symbol-medium-r-normal";

    _PredefinedLineStyleTable["solid"] = 0;
    _PredefinedLineStyleTable["dot"] = CreateLineStyleFromPattern("1100");
    _PredefinedLineStyleTable["dash"] = CreateLineStyleFromPattern("11111100");
    _PredefinedLineStyleTable["dash"] = CreateLineStyleFromPattern("1111111111110000");
    _PredefinedLineStyleTable["longdash"] = CreateLineStyleFromPattern("1111111111111100");
    _PredefinedLineStyleTable["dashdot"] = CreateLineStyleFromPattern("11111111111100111100");
    _PredefinedLineStyleTable["dashdotdot"] = CreateLineStyleFromPattern("1111111111110011001100");
}

TKinokoCanvasImageAreaGtk::~TKinokoCanvasImageAreaGtk()
{
    delete[] _PointListBuffer;

    gdk_color_free(_ColorList[0]);
    gdk_color_free(_ColorList[1]);
    for (unsigned i = 2; i < _ColorList.size(); i++) {
	delete _ColorList[i];
    }

    for (unsigned i = 0; i < _LineStyleList.size(); i++) {
	delete _LineStyleList[i].first;
    }

    for (unsigned i = 0; i < _ImageList.size(); i++) {
	delete _ImageList[i];
    }

    for (unsigned i = 0; i < _PixmapList.size(); i++) {
	gdk_pixmap_unref(_PixmapList[i].first);
	gdk_bitmap_unref(_PixmapList[i].second);
    }
}

void TKinokoCanvasImageAreaGtk::Redraw(void)
{
    gdk_draw_pixmap(
	_DrawingArea->window, _BlackGC,	_Drawable, 0, 0, 
	0, 0, _Width, _Height
    );
}

void TKinokoCanvasImageAreaGtk::Clear(void)
{
    gdk_draw_rectangle(_Drawable, _WhiteGC, TRUE, 0, 0, _Width, _Height);
}

void TKinokoCanvasImageAreaGtk::ClearArea(float x0, float y0, float x1, float y1)
{
    gdk_draw_rectangle(
	_Drawable, _WhiteGC, TRUE,
	min((int) x0, (int) x1), min((int) y0, (int) y1), 
	abs((int) x1 - (int) x0), abs((int) y1 - (int) y0)
    );
}

int TKinokoCanvasImageAreaGtk::SaveImageTo(const string& FileName)
{
    GdkImlibImage* ImlibImage = gdk_imlib_create_image_from_drawable(
	_Drawable, NULL, 0, 0, _Width, _Height
    );

    char FileNameBuffer[512];
    strcpy(FileNameBuffer, FileName.c_str());
    int Result = gdk_imlib_save_image(ImlibImage, FileNameBuffer, NULL);

    return Result;
}

GdkPoint* TKinokoCanvasImageAreaGtk::GdkPointList(const vector<pair<float, float> >& PointList)
{
    int NumberOfPoints = PointList.size();
    if ((_PointListBuffer == 0) || (_PointListBufferSize < NumberOfPoints)) {
	_PointListBufferSize = NumberOfPoints * 2;	
	delete[] _PointListBuffer;
	_PointListBuffer = new GdkPoint[_PointListBufferSize];
    }

    for (int i = 0; i < NumberOfPoints; i++) {
	_PointListBuffer[i].x = (int) PointList[i].first;
	_PointListBuffer[i].y = (int) PointList[i].second;
    }

    return _PointListBuffer;
}

void TKinokoCanvasImageAreaGtk::DrawPoint(float x, float y)
{
    gdk_draw_point(_Drawable, _BlackGC, (int) x, (int) y);
}

void TKinokoCanvasImageAreaGtk::DrawLine(float x0, float y0, float x1, float y1)
{
    gdk_draw_line(_Drawable, _BlackGC, (int) x0, (int) y0, (int) x1, (int) y1);
}

void TKinokoCanvasImageAreaGtk::DrawRect(float x0, float y0, float x1, float y1)
{
    gdk_draw_rectangle(
	_Drawable, _BlackGC, FALSE, 
	min((int) x0, (int) x1), min((int) y0, (int) y1), abs((int) x1 - (int) x0), abs((int) y1 - (int) y0)
    );
}

void TKinokoCanvasImageAreaGtk::DrawRectFill(float x0, float y0, float x1, float y1)
{
    gdk_draw_rectangle(
	_Drawable, _BlackGC, TRUE, 
	min((int) x0, (int) x1), min((int) y0, (int) y1), abs((int) x1 - (int) x0), abs((int) y1 - (int) y0)
    );
}

void TKinokoCanvasImageAreaGtk::DrawLines(const vector<pair<float, float> >& PointList)
{
    GdkPoint* GdkPoint = GdkPointList(PointList);
    gdk_draw_lines(_Drawable, _BlackGC, GdkPoint, PointList.size());
}

void TKinokoCanvasImageAreaGtk::DrawPolygon(const vector<pair<float, float> >& PointList)
{
    GdkPoint* GdkPoint = GdkPointList(PointList);
    gdk_draw_polygon(_Drawable, _BlackGC, FALSE, GdkPoint, PointList.size());
}

void TKinokoCanvasImageAreaGtk::DrawPolygonFill(const vector<pair<float, float> >& PointList)
{
    GdkPoint* GdkPoint = GdkPointList(PointList);
    gdk_draw_polygon(_Drawable, _BlackGC, TRUE, GdkPoint, PointList.size());
}

void TKinokoCanvasImageAreaGtk::DrawCircle(float x, float y, float r)
{
    gdk_draw_arc(
	_Drawable, _BlackGC, FALSE, 
	(int) (x - r), (int) (y - r), (int) (r * 2), (int) (r * 2), 0, (360 * 64)
    );
}

void TKinokoCanvasImageAreaGtk::DrawCircleFill(float x, float y, float r)
{
    gdk_draw_arc(
	_Drawable, _BlackGC, TRUE, 
	(int) (x - r), (int) (y - r), (int) (r * 2), (int) (r * 2), 0, (360 * 64)
    );
}

void TKinokoCanvasImageAreaGtk::DrawEllipse(float x, float y, float rx, float ry, float Angle)
{
    gdk_draw_arc(
	_Drawable, _BlackGC, FALSE, 
	(int) (x - rx), (int) (y - ry), (int) (rx * 2), (int) (ry * 2), 0, (360 * 64)
    );
}

void TKinokoCanvasImageAreaGtk::DrawEllipseFill(float x, float y, float rx, float ry, float Angle)
{
    gdk_draw_arc(
	_Drawable, _BlackGC, TRUE, 
	(int) (x - rx), (int) (y - ry), (int) (rx * 2), (int) (ry * 2), 0, (360 * 64)
    );
}

void TKinokoCanvasImageAreaGtk::DrawText(float x, float y, const string& Text, const string& Adjustment)
{
    if (Text.empty()) {
	return;
    }

    if (! Adjustment.empty()) {
	_CurrentTextAdjustment = SetTextAdjustment(Adjustment);
    }

    if (_CurrentTextOrientation != 0) {
	DrawTextRotated(x, y, Text, _CurrentTextOrientation);
	return;
    }

    if (_VerticalTextAdjustment == TextAdjustment_Top) {
	y += TextAscentOf(Text);
    }
    else if (_VerticalTextAdjustment == TextAdjustment_Center) {
	y += TextAscentOf(Text) / 2;
    }
    if (_HorizontalTextAdjustment == TextAdjustment_Right) {
	x -= TextWidthOf(Text);
    }
    else if (_HorizontalTextAdjustment == TextAdjustment_Center) {
	x -= TextWidthOf(Text) / 2;
    }

    gdk_draw_string(_Drawable, _CurrentFont, _BlackGC, (int) x, (int) y, Text.c_str());
}

int TKinokoCanvasImageAreaGtk::AllocateColor(const std::string& ColorName)
{
    if (_PredefinedColorTable.count(ColorName) > 0) {
        return _PredefinedColorTable[ColorName];
    }
    if (_ColorCacheTable.count(ColorName) > 0) {
	return _ColorCacheTable[ColorName];
    }

    int ColorIndex = -1;

    size_t Separator;
    if ((Separator = ColorName.find_first_of(":")) != string::npos) {
        string TypeName = ColorName.substr(0, Separator);
	string Parameter = ColorName.substr(Separator + 1, string::npos);
	
	if (TypeName == "rgb") {
	    long RgbValue = -1;
	    istringstream RgbValueStream(Parameter);
	    RgbValueStream >> hex >> RgbValue;
	    if (RgbValue >= 0) {
	        float Red = ((RgbValue >> 16) & 0xff) / 256.0;
		float Green = ((RgbValue >> 8) & 0xff) / 256.0;
		float Blue = ((RgbValue >> 0) & 0xff) / 256.0;
		
		ColorIndex = AllocateColorRgb(Red, Green, Blue);
	    }
	}
	else if (TypeName == "hsv") {
	    long HsvValue = -1;
	    istringstream HsvValueStream(Parameter);
	    HsvValueStream >> hex >> HsvValue;
	    if (HsvValue >= 0) {
	        float Hue = ((HsvValue >> 16) & 0xff) / 256.0;
		float Saturation = ((HsvValue >> 8) & 0xff) / 256.0;
		float Brightness = ((HsvValue >> 0) & 0xff) / 256.0;
		
		float Red, Green, Blue;
		TKinokoCanvasRainbowColorScale::HSV2RGB(
	            Hue, Saturation, Brightness, Red, Green, Blue
		);
		
		ColorIndex = AllocateColorRgb(Red, Green, Blue);
	    }
	}
	else if (TypeName == "rainbow") {
	    float Value;
	    istringstream ValueStream(Parameter);
	    if (ValueStream >> Value) {
	        float Hue = 0.7 * Value;
		float Saturation = 1.0;
		float Brightness = 0.5;
		
		float Red, Green, Blue;
		TKinokoCanvasRainbowColorScale::HSV2RGB(
		    Hue, Saturation, Brightness, Red, Green, Blue
		);
		
		ColorIndex = AllocateColorRgb(Red, Green, Blue);
	    }
	}
	else if (TypeName == "grayscale") {
	    float Value;
	    istringstream ValueStream(Parameter);
	    if (ValueStream >> Value) {
	        float Red = 1.0 - Value;
		float Green = 1.0 - Value;
		float Blue = 1.0 - Value;
		
		ColorIndex = AllocateColorRgb(Red, Green, Blue);
	    }
	}
    }

    if (ColorIndex >= 0) {
	_ColorCacheTable[ColorName] = ColorIndex;
    }

    return ColorIndex;
}

int TKinokoCanvasImageAreaGtk::AllocateColorRgb(float Red, float Green, float Blue)
{
    GdkColor* Color = new GdkColor();
    Color->red = (int) (Red * 0xffff);
    Color->green = (int) (Green * 0xffff);
    Color->blue = (int) (Blue * 0xffff);

    gdk_color_alloc(gdk_colormap_get_system(), Color);

    int ColorIndex = _ColorList.size();
    _ColorList.push_back(Color);

    return ColorIndex;
}

int TKinokoCanvasImageAreaGtk::SetColor(int ColorIndex)
{
    if (ColorIndex == _CurrentColorIndex) {
	return _CurrentColorIndex;
    }

    int OldColorIndex = _CurrentColorIndex;

    if ((ColorIndex >= 0) && (ColorIndex < (int) _ColorList.size())) {
	gdk_gc_set_foreground(_BlackGC, _ColorList[ColorIndex]);
	_CurrentColorIndex = ColorIndex;
    }
    
    return OldColorIndex;
}

int TKinokoCanvasImageAreaGtk::SetBackgroundColor(int ColorIndex)
{
    if (
	(ColorIndex < 0) || (ColorIndex >= (int) _ColorList.size()) ||
	(ColorIndex == _CurrentBackgroundColorIndex)
    ){
	return _CurrentBackgroundColorIndex;
    }

    int OldColorIndex = _CurrentBackgroundColorIndex;
    gdk_gc_set_foreground(_WhiteGC, _ColorList[ColorIndex]);
    _CurrentBackgroundColorIndex = ColorIndex;
    
    return OldColorIndex;
}

bool TKinokoCanvasImageAreaGtk::GetColor(int ColorIndex, TKinokoCanvasColor& Color)
{
    if ((ColorIndex < 0) || (ColorIndex >= (int) _ColorList.size())) {
	return false;
    }

    float Red = _ColorList[ColorIndex]->red / (float) 0xffff;
    float Green = _ColorList[ColorIndex]->green / (float) 0xffff;
    float Blue = _ColorList[ColorIndex]->blue / (float) 0xffff;
    Color = TKinokoCanvasColor(Red, Green, Blue);

    return true;
}

int TKinokoCanvasImageAreaGtk::SetLineWidth(int LineWidth)
{
    if (LineWidth == _CurrentLineWidth) {
	return _CurrentLineWidth;
    }

    int OldLineWidth = _CurrentLineWidth;
    _CurrentLineWidth = LineWidth;

    gdk_gc_set_line_attributes(
	_BlackGC, _CurrentLineWidth, _CurrentLineSolidness, 
	GDK_CAP_BUTT, GDK_JOIN_MITER
    );

    return OldLineWidth;
}

int TKinokoCanvasImageAreaGtk::CreateLineStyle(const string& LineStyleName)
{
    int LineStyleIndex = -1;

    if (_PredefinedLineStyleTable.count(LineStyleName) > 0) {
	LineStyleIndex = _PredefinedLineStyleTable[LineStyleName];
    }
    else {
	size_t Separator;
	if ((Separator = LineStyleName.find_first_of(":")) != string::npos) {
	    string TypeName = LineStyleName.substr(0, Separator);
	    string Parameter = LineStyleName.substr(Separator + 1, string::npos);

	    if (TypeName == "pattern") {
		LineStyleIndex = CreateLineStyleFromPattern(Parameter);
	    }
	}
    }

    return LineStyleIndex;
}

int TKinokoCanvasImageAreaGtk::CreateLineStyleFromPattern(const string& LineStylePattern)
{
    if (LineStylePattern.empty()) {
	return 0;
    }

    if (_LineStyleCache.count(LineStylePattern) > 0) {
	return _LineStyleCache[LineStylePattern];
    }

    signed char* LineStyle = new signed char [LineStylePattern.size()];
    int LineStyleLength = 0;
    LineStyle[LineStyleLength] = 0;

    bool IsLastDotFilled = true;
    for (unsigned i = 0; i < LineStylePattern.size(); i++) {
	bool IsThisDotFilled = ! (
	    (LineStylePattern[i] == '0') || (LineStylePattern[i] == ' ')
	);

	if (IsLastDotFilled == IsThisDotFilled) {
	    LineStyle[LineStyleLength] += 1;
	}
	else {
	    LineStyleLength++;
	    LineStyle[LineStyleLength] = 1;
	    IsLastDotFilled = IsThisDotFilled;
	}
    }
    LineStyleLength++;

    if (LineStyleLength <= 1) {
	return 0;
    }

    int LineStyleIndex = _LineStyleList.size();
    _LineStyleList.push_back(make_pair(LineStyle, LineStyleLength));
    _LineStyleCache[LineStylePattern] = LineStyleIndex;

    return LineStyleIndex;
}

int TKinokoCanvasImageAreaGtk::SetLineStyle(int LineStyleIndex, unsigned Offset)
{
    if (
	(LineStyleIndex == _CurrentLineStyleIndex) && 
	(Offset == _CurrentLineOffset)
    ){
	return _CurrentLineStyleIndex;
    }

    if (LineStyleIndex >= (int) _LineStyleList.size()) {
	return _CurrentLineStyleIndex;
    }

    int OldLineStyleIndex = _CurrentLineStyleIndex;

    if (LineStyleIndex <= 0) {
	_CurrentLineStyleIndex = 0;
	_CurrentLineSolidness = GDK_LINE_SOLID;
    }
    else {
	_CurrentLineSolidness = GDK_LINE_ON_OFF_DASH;
	_CurrentLineStyleIndex = LineStyleIndex;
    }
    _CurrentLineOffset = Offset;

    gdk_gc_set_line_attributes(
	_BlackGC, _CurrentLineWidth, _CurrentLineSolidness,
	GDK_CAP_BUTT, GDK_JOIN_MITER
    );

    if (_CurrentLineStyleIndex > 0) {
	signed char* LineStyle = _LineStyleList[LineStyleIndex].first;
	gint LineStyleLength = _LineStyleList[LineStyleIndex].second;
	Offset %= LineStyleLength;

	gdk_gc_set_dashes(_BlackGC, Offset, LineStyle, LineStyleLength);
    }
    
    return OldLineStyleIndex;
}

bool TKinokoCanvasImageAreaGtk::GetLineStyle(int LineStyleIndex, vector<int>& LineStyle)
{
    if (
	(LineStyleIndex < 0) || 
	(LineStyleIndex >= (int) _LineStyleList.size())
    ){
	return false;
    }

    signed char* ThisLineStyle = _LineStyleList[LineStyleIndex].first;
    gint ThisLineStyleLength = _LineStyleList[LineStyleIndex].second;

    for (gint i = 0; i < ThisLineStyleLength; i++) {
	LineStyle.push_back(ThisLineStyle[i]);
    }

    return true;
}

int TKinokoCanvasImageAreaGtk::SetDrawingFunction(int FunctionId)
{
    int OldDrawingFunction = _CurrentDrawingFunction;
    _CurrentDrawingFunction = FunctionId;

    switch (FunctionId) {
      case DrawingFunction_Copy:
	gdk_gc_set_function(_BlackGC, GDK_COPY);
	break;
      case DrawingFunction_Invert:
	gdk_gc_set_function(_BlackGC, GDK_INVERT);
	break;
      case DrawingFunction_Xor:
	gdk_gc_set_function(_BlackGC, GDK_XOR);
	break;
      case DrawingFunction_And:
	gdk_gc_set_function(_BlackGC, GDK_AND);
	break;
      case DrawingFunction_Or:
	gdk_gc_set_function(_BlackGC, GDK_OR);
	break;
      default:
	gdk_gc_set_function(_BlackGC, GDK_COPY);
	break;
    }

    return OldDrawingFunction;
}

int TKinokoCanvasImageAreaGtk::LoadFont(const string& FontName, int FontSize)
{
    int FontIndex = -1;

    if (_PredefinedFontNameTable.count(FontName) > 0) {
	ostringstream FontNameStream;
	FontNameStream << _PredefinedFontNameTable[FontName];
	FontNameStream << "-*-" << FontSize << "-*-*-*-*-*-*-*";
	FontIndex = LoadSystemFont(FontNameStream.str());
    }
    else {
	size_t Separator;
	if ((Separator = FontName.find_first_of(":")) != string::npos) {
	    string TypeName = FontName.substr(0, Separator);
	    string Parameter = FontName.substr(Separator + 1, string::npos);

	    if (TypeName == "x") {
	        // X-Window font //
		FontIndex = LoadSystemFont(Parameter);
	    }
	}
    }

    return FontIndex;
}

int TKinokoCanvasImageAreaGtk::LoadSystemFont(const string& FontName)
{
    if (_FontCache.count(FontName) > 0) {
	return _FontCache[FontName];
    }

    GdkFont* Font = gdk_font_load(FontName.c_str());

    int FontIndex;
    if (Font == NULL) {
	FontIndex = -1;
    }
    else {
	FontIndex = _FontList.size();
    }

    _FontList.push_back(Font);
    _FontCache[FontName] = FontIndex;

    return FontIndex;
}

int TKinokoCanvasImageAreaGtk::SetFont(int FontIndex)
{
    if ((FontIndex < 0) || (FontIndex >= (int) _FontList.size())) {
	return _CurrentFontIndex;
    }

    int OldFontIndex = _CurrentFontIndex;
    _CurrentFont = _FontList[FontIndex];
    _CurrentFontIndex = FontIndex;

    return OldFontIndex;
}

string TKinokoCanvasImageAreaGtk::SetTextAdjustment(const string& Adjustment)
{
    string OldTextAdjustment = _CurrentTextAdjustment;
    _CurrentTextAdjustment = Adjustment;

    _HorizontalTextAdjustment = TextAdjustment_Center;
    _VerticalTextAdjustment = TextAdjustment_Center;

    if (Adjustment.find_first_of('t') != string::npos) {
	_VerticalTextAdjustment = TextAdjustment_Top;
    }
    if (Adjustment.find_first_of('b') != string::npos) {
	_VerticalTextAdjustment = TextAdjustment_Bottom;
    }
    if (Adjustment.find_first_of('l') != string::npos) {
	_HorizontalTextAdjustment = TextAdjustment_Left;
    }
    if (Adjustment.find_first_of('r') != string::npos) {
	_HorizontalTextAdjustment = TextAdjustment_Right;
    }

    return OldTextAdjustment;
}

int TKinokoCanvasImageAreaGtk::SetTextOrientation(int Digree)
{
    int OldTextOrientation = _CurrentTextOrientation;
    _CurrentTextOrientation = Digree;
    
    return OldTextOrientation;
}

int TKinokoCanvasImageAreaGtk::TextHeightOf(const string& Text)
{
    return gdk_string_height(_CurrentFont, Text.c_str());
}

int TKinokoCanvasImageAreaGtk::TextAscentOf(const string& Text)
{
    gint LeftBearing;   // LeftEnd - Origin  (small value)
    gint RightBearing;  // RightEnd - Origin
    gint Width;         // Right - Origin    (Right =~ RightEnd for non-italic)
    gint Ascent;        // Top - BaseLine
    gint Descent;       // BaseLine - Bottom (= Height - Ascent)

    gdk_string_extents(
	_CurrentFont, Text.c_str(), 
	&RightBearing, &LeftBearing, &Width, &Ascent, &Descent
    );

    return Ascent;
}

int TKinokoCanvasImageAreaGtk::TextWidthOf(const string& Text)
{
    return gdk_string_width(_CurrentFont, Text.c_str());
}

TKinokoCanvasImage* TKinokoCanvasImageAreaGtk::LoadXpmImage(const std::string& FileName)
{
    if (_ImageCache.count(FileName) > 0) {
	return _ImageCache[FileName];
    }

    GdkBitmap* PixmapMask;
    GdkPixmap* Pixmap = gdk_pixmap_create_from_xpm(
	_DrawingArea->window, &PixmapMask, 0, FileName.c_str()
    );
    if (Pixmap == NULL) {
	return 0;
    }

    gint Width, Height;
    gdk_window_get_size(Pixmap, &Width, &Height);

    int ImageId = _ImageList.size();
    TKinokoCanvasImage* Image = new TKinokoCanvasImage(ImageId, Width, Height);

    _ImageList.push_back(Image);
    _PixmapList.push_back(make_pair(Pixmap, PixmapMask));
    _ImageCache[FileName] = Image;

    return Image;
}

void TKinokoCanvasImageAreaGtk::DrawImage(float x, float y, TKinokoCanvasImage* Image)
{
    int ImageId = Image->ImageId();
    GdkPixmap* Pixmap = _PixmapList[ImageId].first;
    GdkBitmap* PixmapMask = _PixmapList[ImageId].second;

    gdk_gc_set_clip_mask(_BlackGC, PixmapMask);
    gdk_gc_set_clip_origin(_BlackGC, (int) x, (int) y);

    gdk_draw_pixmap(
	_Drawable, _BlackGC, Pixmap, 0, 0, 
	(int) x, (int) y, Image->Width(), Image->Height()
    );

    gdk_gc_set_clip_mask(_BlackGC, NULL);
}

void TKinokoCanvasImageAreaGtk::DrawTextRotated(float x, float y, const string& Text, int Digree)
{
    gint LeftBearing, RightBearing, TextWidth;
    gint Ascent, Descent;
    gdk_string_extents(
	_CurrentFont, Text.c_str(), 
	&RightBearing, &LeftBearing, &TextWidth, &Ascent, &Descent
    );

    int Width = LeftBearing + RightBearing;
    int Height = Ascent + Descent;

    GdkPixmap* Pixmap = gdk_pixmap_new(
	_DrawingArea->window, Width, Height, -1
    );

    gdk_draw_rectangle(Pixmap, _WhiteGC, TRUE, 0, 0, Width, Height);
    gdk_draw_string(
	Pixmap, _CurrentFont, _BlackGC, 0, Ascent, Text.c_str()
    );

    GdkImlibImage* ImlibImage = gdk_imlib_create_image_from_drawable(
	Pixmap, NULL, 0, 0, Width, Height
    );

    //... 'Digree' is not used, and the rotated image is flipped...
    if (Digree == 90) {
	gdk_imlib_rotate_image(ImlibImage, 90);
	gdk_imlib_flip_image_vertical(ImlibImage);
    }
    else if (Digree == 180) {
	gdk_imlib_flip_image_vertical(ImlibImage);
	gdk_imlib_flip_image_horizontal(ImlibImage);
    }
    else if (Digree == 270) {
	gdk_imlib_rotate_image(ImlibImage, 90);
	gdk_imlib_flip_image_horizontal(ImlibImage);
    }
    else {
	; //...
    }

    float Angle = Digree / 180.0 * 3.14159;
    float NewWidth = (fabs(Width * cos(Angle)) + fabs(Height * sin(Angle)));
    float NewHeight = (fabs(Width * sin(Angle)) + fabs(Height * cos(Angle)));

    gdk_imlib_render(ImlibImage, (int) NewWidth, (int) NewHeight);
    GdkPixmap* NewPixmap = gdk_imlib_move_image(ImlibImage);

    if (_VerticalTextAdjustment == TextAdjustment_Bottom) {
	y -= NewHeight;
    }
    else if (_VerticalTextAdjustment == TextAdjustment_Center) {
	y -= NewHeight / 2;
    }
    if (_HorizontalTextAdjustment == TextAdjustment_Right) {
	x -= NewWidth;
    }
    else if (_HorizontalTextAdjustment == TextAdjustment_Center) {
	x -= NewWidth / 2;
    }

    gdk_draw_pixmap(
	_Drawable, _BlackGC, NewPixmap, 
	0, 0, (int) x, (int) y, (int) NewWidth, (int) NewHeight
    );

    gdk_pixmap_unref(NewPixmap);
    gdk_pixmap_unref(Pixmap);

    //... does ImlibImage need to be freed?
}
