/* KinokoCanvasImageAreaGtkPango.cc */
/* Created by Enomoto Sanshiro on 9 July 2000. */
/* Last updated by Enomoto Sanshiro on 26 December 2009. */


#include <iostream>
#include <sstream>
#include <string>
#include <vector>
#include <map>
#include <cmath>
#include <algorithm>
#include <gtk/gtk.h>
#include <gdk-pixbuf/gdk-pixbuf.h>
#include "KinokoCanvasColorScale.hh"
#include "KinokoCanvasImageAreaGtkPango.hh"

using namespace std;


TKinokoCanvasImageAreaGtkPango::TKinokoCanvasImageAreaGtkPango(GtkWidget* DrawingArea, int Width, int Height)
: TKinokoCanvasImageArea(
    (Width > 0) ? Width : DrawingArea->allocation.width, 
    (Height > 0) ? Height : DrawingArea->allocation.height
) 
{
    _DrawingArea = DrawingArea;
    _Drawable = gdk_pixmap_new(_DrawingArea->window, _Width, _Height, -1);

    _BlackGC = gdk_gc_new(_DrawingArea->window);
    _WhiteGC = gdk_gc_new(_DrawingArea->window);
    gdk_gc_copy(_BlackGC, _DrawingArea->style->fg_gc[0]);
    gdk_gc_copy(_WhiteGC, _DrawingArea->style->bg_gc[0]);

    _PangoContext = gtk_widget_create_pango_context(DrawingArea);
    _PangoLayout = pango_layout_new(_PangoContext);
    pango_layout_set_indent(_PangoLayout, 0);
    pango_layout_set_spacing(_PangoLayout, 0);
    pango_layout_set_auto_dir(_PangoLayout, FALSE);
    pango_layout_set_single_paragraph_mode(_PangoLayout, TRUE);

    _PointListBuffer = 0;
    _PointListBufferSize = 0;

    GdkColor* ForegroundColor = gdk_color_copy(&_DrawingArea->style->fg[0]);
    _ColorList.push_back(ForegroundColor);
    _CurrentColorIndex = 0;
    GdkColor* BackgroundColor = gdk_color_copy(&_DrawingArea->style->bg[0]);
    _ColorList.push_back(BackgroundColor);
    _CurrentBackgroundColorIndex = 1;

    _CurrentLineWidth = 1;

    _CurrentLineStyleIndex = 0;
    _LineStyleList.push_back(pair<signed char*, int>(0, 0));
    _CurrentLineSolidness = GDK_LINE_SOLID;
    _CurrentLineOffset = 0;

    _CurrentDrawingFunction = DrawingFunction_Copy;

    _SystemFont = pango_context_get_font_description(_PangoContext);
    pango_layout_set_font_description(_PangoLayout, _SystemFont);
    _CurrentFontIndex = 0;
    _CurrentFont = _SystemFont;
    _FontList.push_back(_CurrentFont);

    PangoRectangle GlyphSize;
    pango_layout_set_text(_PangoLayout, "0", -1);
    pango_layout_get_pixel_extents(_PangoLayout, &GlyphSize, NULL);
    _CurrentFontAscent = PANGO_DESCENT(GlyphSize);

    _CurrentTextAdjustment = "bl";
    _VerticalTextAdjustment = TextAdjustment_Bottom;
    _HorizontalTextAdjustment = TextAdjustment_Left;
    _CurrentTextOrientation = 0;

    _PangoMatrix = new PangoMatrix();
    _PangoMatrix->xx = 1;
    _PangoMatrix->xy = 0;
    _PangoMatrix->yx = 0;
    _PangoMatrix->yy = 1;
    _PangoMatrix->x0 = 0;
    _PangoMatrix->y0 = 0;

    _PredefinedColorTable["white"] = AllocateColorRgb(1.0, 1.0, 1.0);
    _PredefinedColorTable["black"] = AllocateColorRgb(0, 0, 0);
    _PredefinedColorTable["gray"] = AllocateColorRgb(0.5, 0.5, 0.5);
    _PredefinedColorTable["red"] = AllocateColorRgb(1.0, 0, 0);
    _PredefinedColorTable["green"] = AllocateColorRgb(0, 0.6, 0);
    _PredefinedColorTable["blue"] = AllocateColorRgb(0, 0, 1.0);
    _PredefinedColorTable["yellow"] = AllocateColorRgb(0.9, 0.8, 0);
    _PredefinedColorTable["cyan"] = AllocateColorRgb(0, 0.8, 0.8);
    _PredefinedColorTable["magenta"] = AllocateColorRgb(0.55, 0, 0.55);
    _PredefinedColorTable["orange"] = AllocateColorRgb(1.0, 0.65, 0);
    _PredefinedColorTable["brown"] = AllocateColorRgb(0.54, 0.27, 0.07);
    _PredefinedColorTable["bluegreen"] = AllocateColorRgb(0.13, 0.70, 0.67);
    _PredefinedColorTable["greenyellow"] = AllocateColorRgb(0.68, 1.0, 0.18);
    _PredefinedColorTable["wheat"] = AllocateColorRgb(0.8, 0.67, 0.49);
    _PredefinedColorTable["lightred"] = AllocateColorRgb(1.0, 0.5, 0.5);
    _PredefinedColorTable["lightgreen"] = AllocateColorRgb(0.5, 1.0, 0.5);
    _PredefinedColorTable["lightblue"] = AllocateColorRgb(0.5, 0.5, 1.0);
    _PredefinedColorTable["lightyellow"] = AllocateColorRgb(1.0, 1.0, 0.3);
    _PredefinedColorTable["lightcyan"] = AllocateColorRgb(0.5, 1.0, 1.0);
    _PredefinedColorTable["lightmagenta"] = AllocateColorRgb(1.0, 0.5, 1.0);
    _PredefinedColorTable["lightgray"] = AllocateColorRgb(0.83, 0.83, 0.83);
    _PredefinedColorTable["lightlightyellow"] = AllocateColorRgb(1.0, 1.0, 0.8);
    _PredefinedColorTable["lightlightgray"] = AllocateColorRgb(0.92, 0.92, 0.92);
    _PredefinedColorTable["faintgray"] = AllocateColorRgb(0.98, 0.98, 0.98);
    _PredefinedColorTable["faintred"] = AllocateColorRgb(1.0, 0.96, 0.96);
    _PredefinedColorTable["faintgreen"] = AllocateColorRgb(0.96, 1.0, 0.96);
    _PredefinedColorTable["faintblue"] = AllocateColorRgb(0.96, 0.96, 1.0);
    _PredefinedColorTable["darkgray"] = AllocateColorRgb(0.3, 0.3, 0.3);
    _PredefinedColorTable["darkred"] = AllocateColorRgb(0.63, 0, 0);

    //... for backward compatibility ...//
    _PredefinedFontNameTable["times"] = "times";
    _PredefinedFontNameTable["times-roman"] = "times";
    _PredefinedFontNameTable["times-italic"] = "times italic";
    _PredefinedFontNameTable["times-bold"] = "times bold";
    _PredefinedFontNameTable["times-bolditalic"] = "times bold italic";
    _PredefinedFontNameTable["helvetica"] = "helvetica";
    _PredefinedFontNameTable["helvetica-oblique"] = "helvetica oblique";
    _PredefinedFontNameTable["helvetica-bold"] = "helvetica bold";
    _PredefinedFontNameTable["helvetica-boldoblique"] = "helvetica bold oblique";
    _PredefinedFontNameTable["courier"] = "courier";
    _PredefinedFontNameTable["courier-oblique"] = "courier oblique";
    _PredefinedFontNameTable["courier-bold"] = "courier bold";
    _PredefinedFontNameTable["courier-boldoblique"] = "courier bold oblique";
    _PredefinedFontNameTable["symbol"] = "symbol";

    _PredefinedLineStyleTable["solid"] = 0;
    _PredefinedLineStyleTable["dot"] = CreateLineStyleFromPattern("1100");
    _PredefinedLineStyleTable["dash"] = CreateLineStyleFromPattern("11111100");
    _PredefinedLineStyleTable["dash"] = CreateLineStyleFromPattern("1111111111110000");
    _PredefinedLineStyleTable["longdash"] = CreateLineStyleFromPattern("1111111111111100");
    _PredefinedLineStyleTable["dashdot"] = CreateLineStyleFromPattern("11111111111100111100");
    _PredefinedLineStyleTable["dashdotdot"] = CreateLineStyleFromPattern("1111111111110011001100");
}

TKinokoCanvasImageAreaGtkPango::~TKinokoCanvasImageAreaGtkPango()
{
    delete[] _PointListBuffer;

    g_object_unref(_BlackGC);
    g_object_unref(_WhiteGC);

    gdk_color_free(_ColorList[0]);
    gdk_color_free(_ColorList[1]);
    for (unsigned i = 2; i < _ColorList.size(); i++) {
	delete _ColorList[i];
    }

    g_object_unref(_PangoContext);
    g_object_unref(_PangoLayout);
    for (unsigned i = 1; i < _FontList.size(); i++) {
	pango_font_description_free(_FontList[i]);
    }

    delete _PangoMatrix;

    for (unsigned i = 0; i < _LineStyleList.size(); i++) {
	delete _LineStyleList[i].first;
    }

    for (unsigned i = 0; i < _ImageList.size(); i++) {
	delete _ImageList[i];
    }

    for (unsigned i = 0; i < _PixbufList.size(); i++) {
	gdk_pixbuf_unref(_PixbufList[i]);
    }

    gdk_pixmap_unref(_Drawable);
}

void TKinokoCanvasImageAreaGtkPango::Redraw(void)
{
    gdk_draw_pixmap(
	_DrawingArea->window, _BlackGC,	_Drawable, 0, 0, 
	0, 0, _Width, _Height
    );
}

void TKinokoCanvasImageAreaGtkPango::Clear(void)
{
    gdk_draw_rectangle(_Drawable, _WhiteGC, TRUE, 0, 0, _Width, _Height);
}

void TKinokoCanvasImageAreaGtkPango::ClearArea(float x0, float y0, float x1, float y1)
{
    gdk_draw_rectangle(
	_Drawable, _WhiteGC, TRUE,
	min((int) x0, (int) x1), min((int) y0, (int) y1), 
	abs((int) x1 - (int) x0), abs((int) y1 - (int) y0)
    );
}

int TKinokoCanvasImageAreaGtkPango::SaveImageTo(const string& FileName)
{
    GdkColormap* Colormap = gdk_drawable_get_colormap(_Drawable);
    if (Colormap == NULL) {
	Colormap = gdk_colormap_get_system();
    }

    GdkPixbuf* Pixbuf = gdk_pixbuf_get_from_drawable(
	NULL, _Drawable, Colormap, 0, 0, 0, 0, _Width, _Height
    );

    GError* Error = NULL;
    int Result = gdk_pixbuf_save(
	Pixbuf, FileName.c_str(), "png", &Error, NULL
    );

    gdk_pixbuf_unref(Pixbuf);

    return Result;
}

GdkPoint* TKinokoCanvasImageAreaGtkPango::GdkPointList(const vector<pair<float, float> >& PointList)
{
    int NumberOfPoints = PointList.size();
    if ((_PointListBuffer == 0) || (_PointListBufferSize < NumberOfPoints)) {
	_PointListBufferSize = NumberOfPoints * 2;	
	delete[] _PointListBuffer;
	_PointListBuffer = new GdkPoint[_PointListBufferSize];
    }

    for (int i = 0; i < NumberOfPoints; i++) {
	_PointListBuffer[i].x = (int) PointList[i].first;
	_PointListBuffer[i].y = (int) PointList[i].second;
    }

    return _PointListBuffer;
}

void TKinokoCanvasImageAreaGtkPango::DrawPoint(float x, float y)
{
    gdk_draw_point(_Drawable, _BlackGC, (int) x, (int) y);
}

void TKinokoCanvasImageAreaGtkPango::DrawLine(float x0, float y0, float x1, float y1)
{
    gdk_draw_line(
	_Drawable, _BlackGC, (int) x0, (int) y0, (int) x1, (int) y1
    );
}

void TKinokoCanvasImageAreaGtkPango::DrawRect(float x0, float y0, float x1, float y1)
{
    gdk_draw_rectangle(
	_Drawable, _BlackGC, FALSE, 
	min((int) x0, (int) x1), min((int) y0, (int) y1), 
	abs((int) x1 - (int) x0), abs((int) y1 - (int) y0)
    );
}

void TKinokoCanvasImageAreaGtkPango::DrawRectFill(float x0, float y0, float x1, float y1)
{
    gdk_draw_rectangle(
	_Drawable, _BlackGC, TRUE, 
	min((int) x0, (int) x1), min((int) y0, (int) y1), 
	abs((int) x1 - (int) x0), abs((int) y1 - (int) y0)
    );
}

void TKinokoCanvasImageAreaGtkPango::DrawLines(const vector<pair<float, float> >& PointList)
{
    GdkPoint* GdkPoint = GdkPointList(PointList);
    gdk_draw_lines(_Drawable, _BlackGC, GdkPoint, PointList.size());
}

void TKinokoCanvasImageAreaGtkPango::DrawPolygon(const vector<pair<float, float> >& PointList)
{
    GdkPoint* GdkPoint = GdkPointList(PointList);
    gdk_draw_polygon(_Drawable, _BlackGC, FALSE, GdkPoint, PointList.size());
}

void TKinokoCanvasImageAreaGtkPango::DrawPolygonFill(const vector<pair<float, float> >& PointList)
{
    GdkPoint* GdkPoint = GdkPointList(PointList);
    gdk_draw_polygon(_Drawable, _BlackGC, TRUE, GdkPoint, PointList.size());
}

void TKinokoCanvasImageAreaGtkPango::DrawCircle(float x, float y, float r)
{
    gdk_draw_arc(
	_Drawable, _BlackGC, FALSE, 
	(int) (x - r), (int) (y - r), 
	(int) (r * 2), (int) (r * 2), 
	0, (360 * 64)
    );
}

void TKinokoCanvasImageAreaGtkPango::DrawCircleFill(float x, float y, float r)
{
    gdk_draw_arc(
	_Drawable, _BlackGC, TRUE, 
	(int) (x - r), (int) (y - r), 
	(int) (r * 2), (int) (r * 2), 
	0, (360 * 64)
    );
}

void TKinokoCanvasImageAreaGtkPango::DrawEllipse(float x, float y, float rx, float ry, float Angle)
{
    gdk_draw_arc(
	_Drawable, _BlackGC, FALSE, 
	(int) (x - rx), (int) (y - ry), 
	(int) (rx * 2), (int) (ry * 2), 
	0, (360 * 64)
    );
}

void TKinokoCanvasImageAreaGtkPango::DrawEllipseFill(float x, float y, float rx, float ry, float Angle)
{
    gdk_draw_arc(
	_Drawable, _BlackGC, TRUE, 
	(int) (x - rx), (int) (y - ry), 
	(int) (rx * 2), (int) (ry * 2), 
	0, (360 * 64)
    );
}

void TKinokoCanvasImageAreaGtkPango::DrawText(float x, float y, const string& Text, const string& Adjustment)
{
    if (Text.empty()) {
	return;
    }

    if (! Adjustment.empty()) {
	_CurrentTextAdjustment = SetTextAdjustment(Adjustment);
    }

    pango_layout_set_text(_PangoLayout, Text.c_str(), -1);

    if (_VerticalTextAdjustment == TextAdjustment_Top) {
	;
    }
    else if (_VerticalTextAdjustment == TextAdjustment_Center) {
	y -= _CurrentFontAscent / 2;
    }
    else if (_VerticalTextAdjustment == TextAdjustment_Bottom) {
	y -= _CurrentFontAscent;
    }

    if (_HorizontalTextAdjustment == TextAdjustment_Left) {
	;
    }
    else {
	PangoRectangle LogicalSize;
	pango_layout_get_pixel_extents(_PangoLayout, NULL, &LogicalSize);
	int Width = LogicalSize.width;
	if (_HorizontalTextAdjustment == TextAdjustment_Right) {
	    x -= Width;
	}
	else if (_HorizontalTextAdjustment == TextAdjustment_Center) {
	    x -= Width / 2;
	}
    }

    gdk_draw_layout(_Drawable, _BlackGC, (int) x, (int) y, _PangoLayout);
}

int TKinokoCanvasImageAreaGtkPango::AllocateColor(const std::string& ColorName)
{
    if (_PredefinedColorTable.count(ColorName) > 0) {
	return _PredefinedColorTable[ColorName];
    }
    
    if (_ColorCacheTable.count(ColorName) > 0) {
	return _ColorCacheTable[ColorName];
    }

    int ColorIndex = -1;

    size_t Separator;
    if ((Separator = ColorName.find_first_of(":")) != string::npos) {
	string TypeName = ColorName.substr(0, Separator);
	string Parameter = ColorName.substr(Separator + 1, string::npos);
	
	if (TypeName == "rgb") {
	    long RgbValue = -1;
	    istringstream RgbValueStream(Parameter);
	    RgbValueStream >> hex >> RgbValue;
	    if (RgbValue >= 0) {
		float Red = ((RgbValue >> 16) & 0xff) / 256.0;
		float Green = ((RgbValue >> 8) & 0xff) / 256.0;
		float Blue = ((RgbValue >> 0) & 0xff) / 256.0;
		
		ColorIndex = AllocateColorRgb(Red, Green, Blue);
	    }
	}
	else if (TypeName == "hsv") {
	    long HsvValue = -1;
	    istringstream HsvValueStream(Parameter);
	    HsvValueStream >> hex >> HsvValue;
	    if (HsvValue >= 0) {
		float Hue = ((HsvValue >> 16) & 0xff) / 256.0;
		float Saturation = ((HsvValue >> 8) & 0xff) / 256.0;
		float Brightness = ((HsvValue >> 0) & 0xff) / 256.0;
		
		float Red, Green, Blue;
		TKinokoCanvasRainbowColorScale::HSV2RGB(
		    Hue, Saturation, Brightness, Red, Green, Blue
		);
		
		ColorIndex = AllocateColorRgb(Red, Green, Blue);
	    }
	}
	else if (TypeName == "rainbow") {
	    float Value;
	    istringstream ValueStream(Parameter);
	    if (ValueStream >> Value) {
		float Hue = 0.7 * Value;
		float Saturation = 1.0;
		float Brightness = 0.5;
		
		float Red, Green, Blue;
		TKinokoCanvasRainbowColorScale::HSV2RGB(
		    Hue, Saturation, Brightness, Red, Green, Blue
		);
		
		ColorIndex = AllocateColorRgb(Red, Green, Blue);
	    }
	}
	else if (TypeName == "grayscale") {
	    float Value;
	    istringstream ValueStream(Parameter);
	    if (ValueStream >> Value) {
		float Red = 1.0 - Value;
		float Green = 1.0 - Value;
		float Blue = 1.0 - Value;
		
		ColorIndex = AllocateColorRgb(Red, Green, Blue);
	    }
	}
    }

    if (ColorIndex >= 0) {
	_ColorCacheTable[ColorName] = ColorIndex;
    }

    return ColorIndex;
}

int TKinokoCanvasImageAreaGtkPango::AllocateColorRgb(float Red, float Green, float Blue)
{
    GdkColor* Color = new GdkColor();
    Color->red = (int) (Red * 0xffff);
    Color->green = (int) (Green * 0xffff);
    Color->blue = (int) (Blue * 0xffff);

    gdk_color_alloc(gdk_colormap_get_system(), Color);

    int ColorIndex = _ColorList.size();
    _ColorList.push_back(Color);

    return ColorIndex;
}

int TKinokoCanvasImageAreaGtkPango::SetColor(int ColorIndex)
{
    if (
	(ColorIndex < 0) || (ColorIndex >= (int) _ColorList.size()) ||
	(ColorIndex == _CurrentColorIndex)
    ){
	return _CurrentColorIndex;
    }

    int OldColorIndex = _CurrentColorIndex;
    gdk_gc_set_foreground(_BlackGC, _ColorList[ColorIndex]);
    _CurrentColorIndex = ColorIndex;
    
    return OldColorIndex;
}

int TKinokoCanvasImageAreaGtkPango::SetBackgroundColor(int ColorIndex)
{
    if (
	(ColorIndex < 0) || (ColorIndex >= (int) _ColorList.size()) ||
	(ColorIndex == _CurrentBackgroundColorIndex)
    ){
	return _CurrentBackgroundColorIndex;
    }

    int OldColorIndex = _CurrentBackgroundColorIndex;
    gdk_gc_set_foreground(_WhiteGC, _ColorList[ColorIndex]);
    _CurrentBackgroundColorIndex = ColorIndex;
    
    return OldColorIndex;
}

bool TKinokoCanvasImageAreaGtkPango::GetColor(int ColorIndex, TKinokoCanvasColor& Color)
{
    if ((ColorIndex < 0) || (ColorIndex >= (int) _ColorList.size())) {
	return false;
    }

    float Red = _ColorList[ColorIndex]->red / (float) 0xffff;
    float Green = _ColorList[ColorIndex]->green / (float) 0xffff;
    float Blue = _ColorList[ColorIndex]->blue / (float) 0xffff;
    Color = TKinokoCanvasColor(Red, Green, Blue);

    return true;
}

int TKinokoCanvasImageAreaGtkPango::SetLineWidth(int LineWidth)
{
    if (LineWidth == _CurrentLineWidth) {
	return _CurrentLineWidth;
    }

    int OldLineWidth = _CurrentLineWidth;
    _CurrentLineWidth = LineWidth;

    gdk_gc_set_line_attributes(
	_BlackGC, _CurrentLineWidth, _CurrentLineSolidness, 
	GDK_CAP_BUTT, GDK_JOIN_MITER
    );

    return OldLineWidth;
}

int TKinokoCanvasImageAreaGtkPango::CreateLineStyle(const string& LineStyleName)
{
    int LineStyleIndex = -1;

    if (_PredefinedLineStyleTable.count(LineStyleName) > 0) {
	LineStyleIndex = _PredefinedLineStyleTable[LineStyleName];
    }
    else {
	size_t Separator;
	if ((Separator = LineStyleName.find_first_of(":")) != string::npos) {
	    string TypeName = LineStyleName.substr(0, Separator);
	    string Parameter = LineStyleName.substr(Separator + 1, string::npos);

	    if (TypeName == "pattern") {
		LineStyleIndex = CreateLineStyleFromPattern(Parameter);
	    }
	}
    }

    return LineStyleIndex;
}

int TKinokoCanvasImageAreaGtkPango::CreateLineStyleFromPattern(const string& LineStylePattern)
{
    if (LineStylePattern.empty()) {
	return 0;
    }

    if (_LineStyleCache.count(LineStylePattern) > 0) {
	return _LineStyleCache[LineStylePattern];
    }

    signed char* LineStyle = new signed char [LineStylePattern.size()];
    int LineStyleLength = 0;
    LineStyle[LineStyleLength] = 0;

    bool IsLastDotFilled = true;
    for (unsigned i = 0; i < LineStylePattern.size(); i++) {
	bool IsThisDotFilled = ! (
	    (LineStylePattern[i] == '0') || (LineStylePattern[i] == ' ')
	);

	if (IsLastDotFilled == IsThisDotFilled) {
	    LineStyle[LineStyleLength] += 1;
	}
	else {
	    LineStyleLength++;
	    LineStyle[LineStyleLength] = 1;
	    IsLastDotFilled = IsThisDotFilled;
	}
    }
    LineStyleLength++;

    if (LineStyleLength <= 1) {
	return 0;
    }

    int LineStyleIndex = _LineStyleList.size();
    _LineStyleList.push_back(make_pair(LineStyle, LineStyleLength));
    _LineStyleCache[LineStylePattern] = LineStyleIndex;

    return LineStyleIndex;
}

int TKinokoCanvasImageAreaGtkPango::SetLineStyle(int LineStyleIndex, unsigned Offset)
{
    if (
	(LineStyleIndex == _CurrentLineStyleIndex) && 
	(Offset == _CurrentLineOffset)
    ){
	return _CurrentLineStyleIndex;
    }

    if (LineStyleIndex >= (int) _LineStyleList.size()) {
	return _CurrentLineStyleIndex;
    }

    int OldLineStyleIndex = _CurrentLineStyleIndex;

    if (LineStyleIndex <= 0) {
	_CurrentLineStyleIndex = 0;
	_CurrentLineSolidness = GDK_LINE_SOLID;
    }
    else {
	_CurrentLineSolidness = GDK_LINE_ON_OFF_DASH;
	_CurrentLineStyleIndex = LineStyleIndex;
    }
    _CurrentLineOffset = Offset;

    gdk_gc_set_line_attributes(
	_BlackGC, _CurrentLineWidth, _CurrentLineSolidness,
	GDK_CAP_BUTT, GDK_JOIN_MITER
    );

    if (_CurrentLineStyleIndex > 0) {
	signed char* LineStyle = _LineStyleList[LineStyleIndex].first;
	gint LineStyleLength = _LineStyleList[LineStyleIndex].second;
	Offset %= LineStyleLength;

	gdk_gc_set_dashes(_BlackGC, Offset, LineStyle, LineStyleLength);
    }
    
    return OldLineStyleIndex;
}

bool TKinokoCanvasImageAreaGtkPango::GetLineStyle(int LineStyleIndex, vector<int>& LineStyle)
{
    if (
	(LineStyleIndex < 0) || 
	(LineStyleIndex >= (int) _LineStyleList.size())
    ){
	return false;
    }

    signed char* ThisLineStyle = _LineStyleList[LineStyleIndex].first;
    gint ThisLineStyleLength = _LineStyleList[LineStyleIndex].second;

    for (gint i = 0; i < ThisLineStyleLength; i++) {
	LineStyle.push_back(ThisLineStyle[i]);
    }

    return true;
}

int TKinokoCanvasImageAreaGtkPango::SetDrawingFunction(int FunctionId)
{
    int OldDrawingFunction = _CurrentDrawingFunction;
    _CurrentDrawingFunction = FunctionId;

    switch (FunctionId) {
      case DrawingFunction_Copy:
	gdk_gc_set_function(_BlackGC, GDK_COPY);
	break;
      case DrawingFunction_Invert:
	gdk_gc_set_function(_BlackGC, GDK_INVERT);
	break;
      case DrawingFunction_Xor:
	gdk_gc_set_function(_BlackGC, GDK_XOR);
	break;
      case DrawingFunction_And:
	gdk_gc_set_function(_BlackGC, GDK_AND);
	break;
      case DrawingFunction_Or:
	gdk_gc_set_function(_BlackGC, GDK_OR);
	break;
      default:
	gdk_gc_set_function(_BlackGC, GDK_COPY);
	break;
    }

    return OldDrawingFunction;
}

int TKinokoCanvasImageAreaGtkPango::LoadFont(const string& FontName, int FontSize)
{
    ostringstream s;
    s << FontSize;
    string FontSizeString  = s.str();

    string PangoFontName;
    if (_PredefinedFontNameTable.count(FontName) > 0) {
	PangoFontName = (
            _PredefinedFontNameTable[FontName] + " " + FontSizeString
	);
    }
    else {
	PangoFontName = FontName;
	if (! isdigit(*(PangoFontName.end()-1))) {
	    PangoFontName += " " + FontSizeString;
	}
    }
    
    int FontIndex = _FontList.size();

    _FontList.push_back(
	pango_font_description_from_string(PangoFontName.c_str())
    );

    return FontIndex;
}

int TKinokoCanvasImageAreaGtkPango::LoadSymbolFont(int FontSize)
{
    //... FIXME: use the same font as current font ...//
    return LoadFont("times", FontSize);
}

int TKinokoCanvasImageAreaGtkPango::SetFont(int FontIndex)
{
    if ((FontIndex < 0) || (FontIndex >= (int) _FontList.size())) {
	return _CurrentFontIndex;
    }

    int OldFontIndex = _CurrentFontIndex;
    _CurrentFont = _FontList[FontIndex];
    _CurrentFontIndex = FontIndex;

    //... FIXME: detect font loding errors ...//
    //... TODO: if not found try finding larger/smaller alternatives ...//
    pango_layout_set_font_description(_PangoLayout, _CurrentFont);

    PangoRectangle GlyphSize;
    pango_layout_set_text(_PangoLayout, "0", -1);
    pango_layout_get_pixel_extents(_PangoLayout, &GlyphSize, NULL);
    _CurrentFontAscent = PANGO_DESCENT(GlyphSize);

    return OldFontIndex;
}

string TKinokoCanvasImageAreaGtkPango::SetTextAdjustment(const string& Adjustment)
{
    string OldTextAdjustment = _CurrentTextAdjustment;
    _CurrentTextAdjustment = Adjustment;

    _HorizontalTextAdjustment = TextAdjustment_Center;
    _VerticalTextAdjustment = TextAdjustment_Center;

    if (Adjustment.find_first_of('t') != string::npos) {
	_VerticalTextAdjustment = TextAdjustment_Top;
    }
    if (Adjustment.find_first_of('b') != string::npos) {
	_VerticalTextAdjustment = TextAdjustment_Bottom;
    }
    if (Adjustment.find_first_of('l') != string::npos) {
	_HorizontalTextAdjustment = TextAdjustment_Left;
    }
    if (Adjustment.find_first_of('r') != string::npos) {
	_HorizontalTextAdjustment = TextAdjustment_Right;
    }

    return OldTextAdjustment;
}

int TKinokoCanvasImageAreaGtkPango::SetTextOrientation(int Degree)
{
    if (_CurrentTextOrientation == Degree) {
	return _CurrentTextOrientation;
    }
    int OldTextOrientation = _CurrentTextOrientation;
    _CurrentTextOrientation = Degree;

    if (Degree == 0) {
	pango_context_set_matrix(_PangoContext, NULL);
    }
    else {
	double SinTheta = sin(M_PI * Degree/180.0);
	double CosTheta = cos(M_PI * Degree/180.0);
	_PangoMatrix->xx = CosTheta;
	_PangoMatrix->xy = SinTheta;
	_PangoMatrix->yx = -SinTheta;
	_PangoMatrix->yy = CosTheta;
	_PangoMatrix->x0 = 0;
	_PangoMatrix->y0 = 0;
	pango_context_set_matrix(_PangoContext, _PangoMatrix);
    }
    pango_layout_context_changed(_PangoLayout);
    
    return OldTextOrientation;
}

TKinokoCanvasImage* TKinokoCanvasImageAreaGtkPango::LoadImage(const std::string& FileName)
{
    if (_ImageCache.count(FileName) > 0) {
	return _ImageCache[FileName];
    }

    GError* Error = NULL;
    GdkPixbuf* Pixbuf = gdk_pixbuf_new_from_file(FileName.c_str(), &Error);
    if (Pixbuf == NULL) {
	cerr << "GDK-Pixbuf: " << Error->message << endl;
	return 0;
    }

    int Width = gdk_pixbuf_get_width(Pixbuf);
    int Height = gdk_pixbuf_get_height(Pixbuf);

    int ImageId = _ImageList.size();
    TKinokoCanvasImage* Image = new TKinokoCanvasImage(ImageId, Width, Height);

    _ImageList.push_back(Image);
    _PixbufList.push_back(Pixbuf);
    _ImageCache[FileName] = Image;

    return Image;
}

void TKinokoCanvasImageAreaGtkPango::DrawImage(float x, float y, TKinokoCanvasImage* Image)
{
    int ImageId = Image->ImageId();
    GdkPixbuf* Pixbuf = _PixbufList[ImageId];

    gdk_draw_pixbuf(
	_Drawable, _BlackGC, Pixbuf, 0, 0, 
	(int) x, (int) y, Image->Width(), Image->Height(),
	GDK_RGB_DITHER_NONE, 0, 0
    );
}

int TKinokoCanvasImageAreaGtkPango::TextHeightOf(const string& Text)
{
    PangoRectangle InkSize;
    pango_layout_set_text(_PangoLayout, Text.c_str(), -1);
    pango_layout_get_pixel_extents(_PangoLayout, &InkSize, NULL);

    return InkSize.height;
}

int TKinokoCanvasImageAreaGtkPango::TextAscentOf(const string& Text)
{
    // absolute value of distance between grif baseline and highest point //
    // note that Pango definition is different from this //

    PangoRectangle InkSize;
    pango_layout_set_text(_PangoLayout, Text.c_str(), -1);
    pango_layout_get_pixel_extents(_PangoLayout, &InkSize, NULL);

    return _CurrentFontAscent + PANGO_ASCENT(InkSize);
}

int TKinokoCanvasImageAreaGtkPango::TextWidthOf(const string& Text)
{
    PangoRectangle LogicalSize;
    pango_layout_set_text(_PangoLayout, Text.c_str(), -1);
    pango_layout_get_pixel_extents(_PangoLayout, NULL, &LogicalSize);

    return LogicalSize.width;
}
