/* KinokoControlPanelGtk.cc */
/* Created by Enomoto Sanshiro on 28 September 2001. */
/* Last updated by Enomoto Sanshiro on 28 September 2001. */


#include <cstring>
#include <iostream>
#include <iomanip>
#include <fstream>
#include <sstream>
#include <string>
#include <vector>
#include <deque>
#include <gtk/gtk.h>
#include "MushFileSystem.hh"
#include "KinokoShellAboutDialogGtk.hh"
#include "KinokoShellConfig.hh"
#include "KinokoShellFileSelectDialogGtk.hh"
#include "KinokoControlGtk.hh"
#include "KinokoControlWidgetGtk.hh"
#include "KinokoControlPanelGtk.hh"

using namespace std;


static TKinokoControl* g_Control;

static gint delete_event_cb(GtkWidget* Widget, GdkEventAny* Event, gpointer Data);
static gint expose_event_cb(GtkWidget* Widget, GdkEventExpose* Event);
static gint button_press_event_cb(GtkWidget* Widget, GdkEventButton* Event, gpointer* Data);
static gint button_release_event_cb(GtkWidget* Widget, GdkEventButton* Event, gpointer* Data);
static gboolean widget_focus_event_cb(GtkWidget* Widget, GdkEventFocus* event, gpointer Data);
static void menu_save_as_cb(void);
static void menu_load_cb(void);
static void menu_quit_cb(void);
static void menu_load_script_cb(void);
static void menu_about_cb(void);
static void widget_clicked_cb(GtkWidget* Widget, gpointer Data);
static void file_select_cb(GtkWidget* Widget, gpointer Data);
static void increment_cb(GtkWidget* Widget, gpointer Data);

    
static GtkItemFactoryEntry MenuItemList[] = {
    {"/_File",                   NULL,         0,               0, "<Branch>"},
    {"/File/",                   NULL,         0,               0, "<Separator>"},
    {"/File/e_Xit",              "<control>X", menu_quit_cb,    0, "<Item>"},
    {"/_Action",                 NULL,         0,               0, "<Branch>"},
    {"/Action/Save Values As...", NULL,        menu_save_as_cb, 0, "<Item>"},
    {"/Action/Load Values...",    NULL,        menu_load_cb,    0, "<Item>"},
    {"/Action/Load Script...",   NULL,         menu_load_script_cb, 0, "<Item>"},
    {"/_Help",                   NULL,         0,               0, "<Branch>"},
    {"/Help/_About...",          NULL,         menu_about_cb,   0, "<Item>"},
    {"/Help/",                   NULL,         0,               0, "<Separator>"}
};
static gint NumberOfMenuItems = sizeof(MenuItemList) / sizeof(MenuItemList[0]);



TKinokoControlPanelGtk::TKinokoControlPanelGtk(TKinokoControl* Control, const std::string& RootPath)
: TKinokoControlPanel(Control, RootPath)
{
    _RootWindow = 0;
    _CurrentFrame = 0;
    _CurrentHBox = 0;
    _CurrentVBox = 0;

    _IsInEntryList = false;
    _IsInButtonList = false;
    _IsInToolBar = false;

    _MenuCount = 1;  // the first one is the "File" menu
    _NotebookCount = 0;

    _CurrentMenu = 0;
    _CheckButtonBox = 0;
    _RadioButtonBox = 0;

    g_Control = Control;

    BuildFontNameTable();
}

TKinokoControlPanelGtk::~TKinokoControlPanelGtk()
{
}

void TKinokoControlPanelGtk::CreateRootWindow(void)
{
    _RootWindow = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title(GTK_WINDOW(_RootWindow), "KinokoControlPanel");
    gtk_widget_realize(_RootWindow);

    GtkWidget* RootVBox = gtk_vbox_new(FALSE, 0);
    gtk_container_add(GTK_CONTAINER(_RootWindow), RootVBox);

    // Menu and Menubar //
    GtkAccelGroup* AccelGroup = gtk_accel_group_new();
    _ItemFactory = gtk_item_factory_new(
	GTK_TYPE_MENU_BAR, "<menu>", AccelGroup
    );
    gtk_item_factory_create_items(
	_ItemFactory, NumberOfMenuItems, MenuItemList, NULL
    );
    gtk_window_add_accel_group(GTK_WINDOW(_RootWindow), AccelGroup);
    GtkWidget* MenuBar = gtk_item_factory_get_widget(_ItemFactory, "<menu>");
    GtkWidget* MenuHandleBox = gtk_handle_box_new();
    gtk_container_add(GTK_CONTAINER(MenuHandleBox), MenuBar);
    gtk_box_pack_start(GTK_BOX(RootVBox), MenuHandleBox, FALSE, FALSE, 0);

    // ToolBar Box //
    _ToolBar = 0;
    _ToolBarHBox = gtk_hbox_new(FALSE, 0);
    gtk_box_pack_start(GTK_BOX(RootVBox), _ToolBarHBox, FALSE, FALSE, 0);

    // Contents Box //
    GtkWidget* RootHBox = gtk_hbox_new(FALSE, 0);
    _CurrentVBox = gtk_vbox_new(FALSE, 0);
    _CurrentHBox = gtk_hbox_new(FALSE, 0);
    gtk_box_pack_start(GTK_BOX(RootVBox), RootHBox, FALSE, FALSE, 3);
    gtk_box_pack_start(GTK_BOX(RootHBox), _CurrentVBox, FALSE, FALSE, 3);
    gtk_box_pack_start(GTK_BOX(_CurrentVBox), _CurrentHBox, FALSE, FALSE, 0);

    gtk_signal_connect(
	GTK_OBJECT(_RootWindow), "delete_event", 
	GTK_SIGNAL_FUNC(delete_event_cb), NULL
    );
}

void TKinokoControlPanelGtk::OpenPanel(map<string, string>& OptionTable)
{
    string Name = OptionTable["name"];
    string Label = OptionTable["label"];

    if (_RootWindow == 0) {
	CreateRootWindow();
	gtk_window_set_title(GTK_WINDOW(_RootWindow), Label.c_str());
    }
}

void TKinokoControlPanelGtk::ClosePanel(void)
{
    gtk_widget_show_all(_RootWindow);
}

void TKinokoControlPanelGtk::OpenFrame(map<string, string>& OptionTable)
{
    string Name = OptionTable["name"];
    string Label = OptionTable["label"];

    _FrameStack.push_front(_CurrentFrame);
    _VBoxStack.push_front(_CurrentVBox);
    _HBoxStack.push_front(_CurrentHBox);

    _CurrentFrame = gtk_frame_new(Label.c_str());
    gtk_box_pack_start(GTK_BOX(_CurrentHBox), _CurrentFrame, TRUE, TRUE, 0);

    _CurrentVBox = gtk_vbox_new(FALSE, 0);
    _CurrentHBox = gtk_hbox_new(FALSE, 0);
    gtk_container_add(GTK_CONTAINER(_CurrentFrame), _CurrentVBox);
    gtk_box_pack_start(GTK_BOX(_CurrentVBox), _CurrentHBox, FALSE, FALSE, 3);

    gtk_container_border_width(GTK_CONTAINER(_CurrentFrame), 3);
}

void TKinokoControlPanelGtk::CloseFrame(void)
{
    _CurrentFrame = _FrameStack.front();
    _CurrentVBox = _VBoxStack.front();
    _CurrentHBox = _HBoxStack.front();

    _FrameStack.pop_front();
    _VBoxStack.pop_front();
    _HBoxStack.pop_front();
}

void TKinokoControlPanelGtk::OpenBox(map<string, string>& OptionTable)
{
    GtkWidget* OldHBox = _CurrentHBox;
    _VBoxStack.push_front(_CurrentVBox);
    _HBoxStack.push_front(_CurrentHBox);

    _CurrentVBox = gtk_vbox_new(FALSE, 0);
    _CurrentHBox = gtk_hbox_new(FALSE, 0);
    gtk_box_pack_start(GTK_BOX(OldHBox), _CurrentVBox, FALSE, FALSE, 0);
    gtk_box_pack_start(GTK_BOX(_CurrentVBox), _CurrentHBox, FALSE, FALSE, 0);
}

void TKinokoControlPanelGtk::CloseBox(void)
{
    _CurrentVBox = _VBoxStack.front();
    _CurrentHBox = _HBoxStack.front();

    _VBoxStack.pop_front();
    _HBoxStack.pop_front();
}

void TKinokoControlPanelGtk::OpenNotebook(std::map<std::string, std::string>& OptionTable)
{
    string Name = OptionTable["name"];
    string Tab = OptionTable["tab"];

    GtkWidget* Notebook = gtk_notebook_new();
    gtk_box_pack_start(GTK_BOX(_CurrentHBox), Notebook, TRUE, TRUE, 0);
    _NotebookStack.push_front(Notebook);

    if (Tab == "bottom") {
	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(Notebook), GTK_POS_BOTTOM);
    }
    else if (Tab == "left") {
	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(Notebook), GTK_POS_LEFT);
    }
    else if (Tab == "right") {
	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(Notebook), GTK_POS_RIGHT);
    }
    else if (Tab == "none") {
	gtk_notebook_set_show_tabs(GTK_NOTEBOOK(Notebook), FALSE);
    }

    gtk_notebook_set_homogeneous_tabs(GTK_NOTEBOOK(Notebook), TRUE);
    gtk_notebook_set_scrollable(GTK_NOTEBOOK(Notebook), TRUE);


    _NotebookCount++;
    if (Name.empty()) {
	ostringstream os;
	os << "notebook_" << setfill('0') << setw(2) << _NotebookCount;
	Name = os.str();
    }

    TKinokoControlWidget* ControlWidget;
    ControlWidget = new TKinokoNotebookWidgetGtk(Name, Notebook);

    _Control->AddWidget(ControlWidget);
    _Control->AddInputWidget(ControlWidget);
}

void TKinokoControlPanelGtk::CloseNotebook(void)
{
    _NotebookStack.pop_front();
}

void TKinokoControlPanelGtk::OpenPage(std::map<std::string, std::string>& OptionTable)
{
    string Label = OptionTable["label"];

    if (Label.empty()) {
	Label = "untitled";
    }
    GtkWidget* LabelWidget = gtk_label_new(Label.c_str());

    _VBoxStack.push_front(_CurrentVBox);
    _HBoxStack.push_front(_CurrentHBox);
    _CurrentVBox = gtk_vbox_new(FALSE, 0);
    _CurrentHBox = gtk_hbox_new(FALSE, 0);

    GtkWidget* Notebook = _NotebookStack.front();
    gtk_notebook_append_page(
	GTK_NOTEBOOK(Notebook), _CurrentVBox, LabelWidget
    );
    gtk_box_pack_start(GTK_BOX(_CurrentVBox), _CurrentHBox, FALSE, FALSE, 3);
}

void TKinokoControlPanelGtk::ClosePage(void)
{
    _CurrentVBox = _VBoxStack.front();
    _CurrentHBox = _HBoxStack.front();

    _VBoxStack.pop_front();
    _HBoxStack.pop_front();
}

void TKinokoControlPanelGtk::OpenTable(std::map<std::string, std::string>& OptionTable)
{
    string NumberOfRowsStr = OptionTable["height"];
    string NumberOfColumnsStr = OptionTable["width"];
    string IsHomogeneousStr = OptionTable["homogeneous"];

    int NumberOfRows = 1;
    istringstream(NumberOfRowsStr) >> NumberOfRows;

    int NumberOfColumns = 1;
    istringstream(NumberOfColumnsStr) >> NumberOfColumns;

    bool IsHomogeneous = (IsHomogeneousStr == "true");

    GtkWidget* Table = gtk_table_new(
	NumberOfRows, NumberOfColumns, IsHomogeneous
    );
    gtk_box_pack_start(GTK_BOX(_CurrentHBox), Table, TRUE, TRUE, 0);

    _TableStack.push_front(Table);
    _TableSizeStack.push_front(make_pair(NumberOfRows, NumberOfColumns));
    _TablePositionStack.push_front(make_pair(0, 0));
}

void TKinokoControlPanelGtk::CloseTable(void)
{
    _TableStack.pop_front();
    _TableSizeStack.pop_front();
    _TablePositionStack.pop_front();
}

void TKinokoControlPanelGtk::OpenCell(std::map<std::string, std::string>& OptionTable)
{
    string TopStr = OptionTable["top"];
    string LeftStr = OptionTable["left"];
    string HeightStr = OptionTable["height"];
    string WidthStr = OptionTable["width"];

    int Top = -1, Left = -1;
    if (! TopStr.empty()) {
	istringstream(TopStr) >> Top;
    }
    if (! LeftStr.empty()) {
	istringstream(LeftStr) >> Left;
    }
    if (Top < 0) {
	Top = _TablePositionStack.front().first;
    }
    if (Left < 0) {
	Left = _TablePositionStack.front().second;
    }

    int Height = 1, Width = 1;
    if (! HeightStr.empty()) {
	istringstream(HeightStr) >> Height;
    }
    if (! WidthStr.empty()) {
	istringstream(WidthStr) >> Width;
    }

    if (
	(Top + Height > _TableSizeStack.front().first) ||
	(Left + Width > _TableSizeStack.front().second)
    ){
	_TableSizeStack.front().first = max(
	    _TableSizeStack.front().first, Top + Height
	);
	_TableSizeStack.front().second = max(
	    _TableSizeStack.front().second, Left + Width
	);
	gtk_table_resize(
	    GTK_TABLE(_TableStack.front()),
	    _TableSizeStack.front().first, _TableSizeStack.front().second
	);
    }

    _VBoxStack.push_front(_CurrentVBox);
    _HBoxStack.push_front(_CurrentHBox);
    _CurrentVBox = gtk_vbox_new(FALSE, 0);
    _CurrentHBox = gtk_hbox_new(FALSE, 0);

    GtkWidget* Table = _TableStack.front();
    gtk_table_attach_defaults(
	GTK_TABLE(Table), _CurrentVBox, Left, Left + Width, Top, Top + Height
    );
    gtk_box_pack_start(GTK_BOX(_CurrentVBox), _CurrentHBox, FALSE, FALSE, 3);
    
    Left += Width;
    if (Left >= _TableSizeStack.front().second) {
	Left = 0;
	Top++;
    }
    _TablePositionStack.front().first = Top;
    _TablePositionStack.front().second = Left;
}

void TKinokoControlPanelGtk::CloseCell(void)
{
    _CurrentVBox = _VBoxStack.front();
    _CurrentHBox = _HBoxStack.front();

    _VBoxStack.pop_front();
    _HBoxStack.pop_front();
}

void TKinokoControlPanelGtk::OpenToolBar(map<string, string>& OptionTable)
{
    if (_ToolBar == 0) {
	_ToolBar = gtk_toolbar_new();
	GtkWidget* HandleBox = gtk_handle_box_new();
	gtk_container_add(GTK_CONTAINER(HandleBox), _ToolBar);
	gtk_box_pack_start(GTK_BOX(_ToolBarHBox), HandleBox, TRUE, TRUE, 0);
    }

    _IsInToolBar = true;
}

void TKinokoControlPanelGtk::CloseToolBar(void)
{
    _IsInToolBar = false;
}

void TKinokoControlPanelGtk::OpenEntryList(map<string, string>& OptionTable)
{
    _IsInEntryList = true;
}

void TKinokoControlPanelGtk::CloseEntryList(void)
{
    int NumberOfEntries = _EntryLabelList.size();

    GtkWidget* TableWidget = gtk_table_new(NumberOfEntries, 3, FALSE);
    for (int i = 0; i < NumberOfEntries; i++) {
	GtkWidget* LabelWidget = _EntryLabelList[i];
	GtkWidget* EntryWidget = _EntryEntryList[i];
	GtkWidget* OptionWidget = _EntryOptionList[i];
	
	GtkWidget* CellHBox = gtk_hbox_new(FALSE, 0);
	GtkWidget* HPadding = gtk_label_new("");
	gtk_box_pack_start(GTK_BOX(CellHBox), LabelWidget, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(CellHBox), HPadding, TRUE, TRUE, 0);

	gtk_table_attach(
	    GTK_TABLE(TableWidget), CellHBox, 
	    0, 1, i, i + 1,
	    GTK_FILL, GTK_SHRINK, 0, 2
	);
	gtk_table_attach(
	    GTK_TABLE(TableWidget), EntryWidget, 
	    1, 2, i, i + 1,
	    GTK_FILL, GTK_SHRINK, 3, 2
	);
	if (OptionWidget != 0) {
	    gtk_table_attach(
		GTK_TABLE(TableWidget), OptionWidget, 
		2, 3, i, i + 1,
		GTK_FILL, GTK_SHRINK, 0, 2
	    );
	}
    }

    gtk_box_pack_start(GTK_BOX(_CurrentHBox), TableWidget, FALSE, FALSE, 3);

    _EntryLabelList.erase(_EntryLabelList.begin(), _EntryLabelList.end());
    _EntryEntryList.erase(_EntryEntryList.begin(), _EntryEntryList.end());
    _EntryOptionList.erase(_EntryOptionList.begin(), _EntryOptionList.end());
    _IsInEntryList = false;
}

void TKinokoControlPanelGtk::AddEntry(map<string, string>& OptionTable)
{
    string Name = OptionTable["name"];
    string Label = OptionTable["label"];
    string Selection = OptionTable["selection"];
    string Width = OptionTable["width"];
    string Alignment = OptionTable["alignment"];
    string Option = OptionTable["option"]; //...
    string OnFocus = OptionTable["on_focus"];
    string OnBlur = OptionTable["on_blur"];

    GtkWidget* LabelWidget = gtk_label_new(Label.c_str());
    GtkWidget* InputWidget;
    GtkWidget* EntryWidget;
    GtkWidget* OptionWidget = 0;
    GtkWidget* ComboWidget = 0;
    GList* ComboItemList = NULL;

    if (Selection.empty()) {
	EntryWidget = gtk_entry_new();
	InputWidget = EntryWidget;
    }
    else {
	string Item;
	istringstream SelectionStream(Selection);
	while (SelectionStream >> Item) {
	    ComboItemList = g_list_append(ComboItemList, strdup(Item.c_str()));
	}

	ComboWidget = gtk_combo_new();
	EntryWidget = GTK_COMBO(ComboWidget)->entry;

	gtk_combo_set_popdown_strings(GTK_COMBO(ComboWidget), ComboItemList);
	gtk_entry_set_editable(GTK_ENTRY(EntryWidget), FALSE);

	InputWidget = ComboWidget;
    }

    if (! Width.empty()) {
	int WidthValue;
	if (istringstream(Width) >> WidthValue) {
	    gtk_widget_set_usize(
		InputWidget, WidthValue, InputWidget->requisition.height
	    );
	}
    }

#if GTK_CHECK_VERSION(2,4,0)
    if (! Alignment.empty()) {
	if (Alignment == "right") {
	    gtk_entry_set_alignment(GTK_ENTRY(EntryWidget), 1.0);
	}
	else if (Alignment == "center") {
	    gtk_entry_set_alignment(GTK_ENTRY(EntryWidget), 0.5);
	}
    }
#endif

    if (Option == "file_select") {
	OptionWidget = gtk_button_new_with_label("select...");
	gtk_signal_connect(
	    GTK_OBJECT(OptionWidget), "clicked", 
	    GTK_SIGNAL_FUNC(file_select_cb), EntryWidget
	);
    }
#if 1 
    //... for backward compatibility ...//
    else if (Option == "increment") {
	OptionWidget = gtk_button_new_with_label("+1");
	gtk_signal_connect(
	    GTK_OBJECT(OptionWidget), "clicked", 
	    GTK_SIGNAL_FUNC(increment_cb), EntryWidget
	);
    }
#endif

    if (_IsInEntryList) {
	_EntryLabelList.push_back(LabelWidget);
	_EntryEntryList.push_back(InputWidget);
	_EntryOptionList.push_back(OptionWidget);
    }
    else {
	gtk_box_pack_start(
	    GTK_BOX(_CurrentHBox), LabelWidget, FALSE, FALSE, 3
	);
	gtk_box_pack_start(
	    GTK_BOX(_CurrentHBox), InputWidget, FALSE, FALSE, 3
	);
	if (OptionWidget != 0) {
	    gtk_box_pack_start(
		GTK_BOX(_CurrentHBox), OptionWidget, FALSE, FALSE, 3
	    );
	}
    }
    
    TKinokoEntryWidget* ControlWidget;
    ControlWidget = new TKinokoEntryWidgetGtk(
	Name, EntryWidget, ComboWidget, ComboItemList
    );

    if (! OnFocus.empty()) {
	ControlWidget->SetActionOnFocus(OnFocus);
	gtk_signal_connect(
	    GTK_OBJECT(EntryWidget), "focus-in-event", 
	    GTK_SIGNAL_FUNC(widget_focus_event_cb), ControlWidget
	);
    }
    if (! OnBlur.empty()) {
	ControlWidget->SetActionOnBlur(OnBlur);
	gtk_signal_connect(
	    GTK_OBJECT(EntryWidget), "focus-out-event", 
	    GTK_SIGNAL_FUNC(widget_focus_event_cb), ControlWidget
	);
    }

    _Control->AddWidget(ControlWidget);
    _Control->AddInputWidget(ControlWidget);
}

void TKinokoControlPanelGtk::OpenButtonList(map<string, string>& OptionTable)
{
    _CurrentHBox = gtk_hbox_new(FALSE, 0);
    gtk_box_pack_start(GTK_BOX(_CurrentVBox), _CurrentHBox, FALSE, FALSE, 3);

    GtkWidget* HPaddingWidget = gtk_label_new("");
    gtk_box_pack_start(GTK_BOX(_CurrentHBox), HPaddingWidget, TRUE, TRUE, 3);

    GtkWidget* ButtonBox = gtk_hbox_new(TRUE, 0);
    gtk_box_pack_start(GTK_BOX(_CurrentHBox), ButtonBox, FALSE, FALSE, 3);
    _CurrentHBox = ButtonBox;
    _IsInButtonList = true;
}

void TKinokoControlPanelGtk::CloseButtonList(void)
{
    _IsInButtonList = false;
}

void TKinokoControlPanelGtk::AddButton(map<string, string>& OptionTable)
{
    string Name = OptionTable["name"];
    string Label = OptionTable["label"];
    string ImageFileName = OptionTable["image"];
    string Tip = OptionTable["tip"];
    string EnabledStateList = OptionTable["enabled_on"];
    string ActionOnClick = OptionTable["on_click"];

    GtkWidget* ButtonWidget = 0;
    if (! ImageFileName.empty()) {
	if (! TMushFileAttribute(ImageFileName).IsReadable()) {
	    string FilePath = TMushFileAttribute(ImageFileName).PathName();
	    if (FilePath.empty() && (! _RootPath.empty())) {
		ImageFileName = _RootPath + ImageFileName;
	    }
	}
	
	GtkWidget* Image = gtk_image_new_from_file(ImageFileName.c_str());
	ButtonWidget = gtk_button_new();
	gtk_container_add(GTK_CONTAINER(ButtonWidget), Image);
    }
    if ((ButtonWidget == 0) && (! Label.empty())) {
	ButtonWidget = gtk_button_new_with_label(Label.c_str());
    }
    if ((ButtonWidget == 0) && (! Name.empty())) {
	ButtonWidget = gtk_button_new_with_label(Name.c_str());
    }
    if (ButtonWidget == 0) {
	ButtonWidget = gtk_button_new_with_label("\?\?\?");
    }

    if (! Tip.empty()) {
	GtkTooltips* ToolTips = gtk_tooltips_new();
	gtk_tooltips_set_tip(ToolTips, ButtonWidget, Tip.c_str(), NULL);
    }

    if (_IsInButtonList) {
	gtk_box_pack_start(
	    GTK_BOX(_CurrentHBox), ButtonWidget, TRUE, TRUE, 2
	);
    }
    else if (_IsInToolBar) {
	gtk_toolbar_append_widget(
	    GTK_TOOLBAR(_ToolBar), ButtonWidget, Tip.c_str(), ""
	);
    }
    else {
	gtk_box_pack_start(
	    GTK_BOX(_CurrentHBox), ButtonWidget, FALSE, FALSE, 3
	);
    }

    if (Name.empty()) {
	Name = ActionOnClick;
    }

    TKinokoControlWidget* ControlWidget;
    ControlWidget = new TKinokoButtonWidgetGtk(
	Name, ButtonWidget, ActionOnClick
    );

    gtk_signal_connect(
	GTK_OBJECT(ButtonWidget), "clicked", 
	GTK_SIGNAL_FUNC(widget_clicked_cb), ControlWidget
    );

    _Control->AddWidget(ControlWidget, EnabledStateList);
}

void TKinokoControlPanelGtk::OpenCheckButtonList(map<string, string>& OptionTable)
{
    _CheckButtonBox = gtk_vbox_new(FALSE, 0);
    gtk_box_pack_start(GTK_BOX(_CurrentHBox), _CheckButtonBox, FALSE, FALSE, 2);
}

void TKinokoControlPanelGtk::CloseCheckButtonList(void)
{
    _CheckButtonBox = 0;
}

void TKinokoControlPanelGtk::AddCheckButton(map<string, string>& OptionTable)
{
    string Name = OptionTable["name"];
    string Label = OptionTable["label"];

    GtkWidget* ButtonWidget = gtk_check_button_new_with_label(Label.c_str());

    if (_CheckButtonBox != 0) {
	gtk_box_pack_start(
	    GTK_BOX(_CheckButtonBox), ButtonWidget, FALSE, FALSE, 0
	);
    }
    else {
	gtk_box_pack_start(
	    GTK_BOX(_CurrentHBox), ButtonWidget, FALSE, FALSE, 0
	);
    }

    TKinokoControlWidget* ControlWidget;
    ControlWidget = new TKinokoCheckButtonWidgetGtk(Name, ButtonWidget);

    _Control->AddWidget(ControlWidget);
    _Control->AddInputWidget(ControlWidget);
}

void TKinokoControlPanelGtk::OpenRadioButtonList(map<string, string>& OptionTable)
{
    string Name = OptionTable["name"];

    _RadioButtonBox = gtk_vbox_new(FALSE, 0);
    gtk_box_pack_start(GTK_BOX(_CurrentHBox), _RadioButtonBox, FALSE, FALSE, 2);
    _RadioButtonGroup = NULL;
    _RadioButtonSet = new TKinokoRadioButtonSetWidget(Name);
}

void TKinokoControlPanelGtk::CloseRadioButtonList(void)
{
    _Control->AddWidget(_RadioButtonSet);
    _Control->AddInputWidget(_RadioButtonSet);

    _RadioButtonBox = 0;
    _RadioButtonSet = 0;
}

void TKinokoControlPanelGtk::AddRadioButton(map<string, string>& OptionTable)
{
    string Name = OptionTable["name"];
    string Label = OptionTable["label"];

    if (_RadioButtonBox == 0) {
	AddCheckButton(OptionTable);
	return;
    }

    GtkWidget* ButtonWidget = gtk_radio_button_new_with_label(
	_RadioButtonGroup, Label.c_str()
    );
    _RadioButtonGroup = gtk_radio_button_group(GTK_RADIO_BUTTON(ButtonWidget));

    gtk_box_pack_start(
	GTK_BOX(_RadioButtonBox), ButtonWidget, FALSE, FALSE, 0
    );

    _RadioButtonSet->AddRadioButton(
	new TKinokoRadioButtonWidgetGtk(Name, ButtonWidget)
    );
}

void TKinokoControlPanelGtk::OpenMenu(std::map<std::string, std::string>& OptionTable)
{
    string Label = OptionTable["label"];

    string Path = "/"+ Label;
    GtkWidget* Menu = gtk_item_factory_get_widget(_ItemFactory, Path.c_str());

    if (Menu == 0) {
	Menu = gtk_menu_new();

	GtkWidget* MenuBar = gtk_item_factory_get_widget(_ItemFactory, "<menu>");
	GtkWidget* MenuItem = gtk_menu_item_new_with_label(Label.c_str());
	gtk_menu_item_set_submenu(GTK_MENU_ITEM(MenuItem), Menu);
	gtk_menu_bar_insert(GTK_MENU_BAR(MenuBar), MenuItem, _MenuCount++);
	gtk_widget_show(MenuItem);
    }

    _CurrentMenu = Menu;
}

void TKinokoControlPanelGtk::CloseMenu(void)
{
    _CurrentMenu = 0;
}

void TKinokoControlPanelGtk::AddMenuItem(std::map<std::string, std::string>& OptionTable)
{
    string Name = OptionTable["name"];
    string Label = OptionTable["label"];
    string EnabledStateList = OptionTable["enabled_on"];
    string ActionOnClick = OptionTable["on_click"];

    if (_CurrentMenu == 0) {
	cerr << "KCML ERROR: menu item outside menu: " << Name << endl;
	return;
    }

    if (Label.empty()) {
	Label = Name;
    }

    GtkWidget* MenuItem = gtk_menu_item_new_with_label(Label.c_str());
    gtk_menu_append(GTK_MENU(_CurrentMenu), MenuItem);
    gtk_widget_show(MenuItem);

    TKinokoControlWidget* ControlWidget;
    ControlWidget = new TKinokoMenuItemWidgetGtk(
	Name, MenuItem, ActionOnClick
    );

    gtk_signal_connect(
	GTK_OBJECT(MenuItem), "activate", 
	GTK_SIGNAL_FUNC(widget_clicked_cb), ControlWidget
    );

    _Control->AddWidget(ControlWidget, EnabledStateList);
}

void TKinokoControlPanelGtk::AddLabel(map<string, string>& OptionTable)
{
    string Name = OptionTable["name"];
    string Label = OptionTable["label"];
    string FontName = OptionTable["font"];
    string FontSize = OptionTable["fontsize"];

#if 1
    //... for backward compatibility ...//
    if (FontSize.empty() && ! OptionTable["size"].empty()) {
	FontSize = OptionTable["size"];
    }
#endif    

    GtkWidget* LabelWidget = gtk_label_new(Label.c_str());
    gtk_box_pack_start(GTK_BOX(_CurrentHBox), LabelWidget, FALSE, FALSE, 3);

    if (! FontName.empty() || ! FontSize.empty()) {
        PangoFontDescription* Font = LoadFont(FontName, FontSize);
	if (Font != NULL) {
	    GtkStyle* Style = gtk_widget_get_style(LabelWidget);
	    GtkStyle* NewStyle = gtk_style_copy(Style);
            NewStyle->font_desc = Font;
            gtk_style_set_font(NewStyle, NULL);

	    gtk_widget_set_style(LabelWidget, NewStyle);
	}
    }

    _Control->AddWidget(new TKinokoLabelWidgetGtk(Name, LabelWidget));
}

void TKinokoControlPanelGtk::AddTextBox(std::map<std::string, std::string>& OptionTable)
{
    string Name = OptionTable["name"];
    string Width = OptionTable["width"];
    string Height = OptionTable["height"];
    string VScroll = OptionTable["vscroll"];
    string HScroll = OptionTable["hscroll"];

    GtkWidget* TextWidget = gtk_text_view_new();

    if (! Width.empty() || ! Height.empty()) {
        int WidthValue = -1, HeightValue = -1;
	istringstream(Width) >> WidthValue;
	istringstream(Height) >> HeightValue;
	gtk_widget_set_usize(TextWidget, WidthValue, HeightValue);
    }

    GtkPolicyType VScrollPolicy, HScrollPolicy;
    if (VScroll == "on") {
	VScrollPolicy = GTK_POLICY_ALWAYS;
    }
    else if (VScroll == "automatic") {
	VScrollPolicy = GTK_POLICY_AUTOMATIC;
    }
    else {
	VScrollPolicy = GTK_POLICY_NEVER;
    }
    if (HScroll == "on") {
	HScrollPolicy = GTK_POLICY_ALWAYS;
    }
    else if (HScroll == "automatic") {
	HScrollPolicy = GTK_POLICY_AUTOMATIC;
    }
    else {
	HScrollPolicy = GTK_POLICY_NEVER;
    }

    GtkWidget* ScrolledWindow = gtk_scrolled_window_new(NULL, NULL);
    gtk_scrolled_window_set_policy(
	GTK_SCROLLED_WINDOW(ScrolledWindow), HScrollPolicy, VScrollPolicy
    );

    gtk_container_add(GTK_CONTAINER(ScrolledWindow), TextWidget);
    gtk_box_pack_start(GTK_BOX(_CurrentHBox), ScrolledWindow, TRUE, TRUE, 3);

    TKinokoControlWidget* ControlWidget;
    ControlWidget = new TKinokoTextBoxWidgetGtk(Name, TextWidget);

    _Control->AddWidget(ControlWidget);
    _Control->AddInputWidget(ControlWidget);
}

void TKinokoControlPanelGtk::AddImage(map<string, string>& OptionTable)
{
    string ImageFileName = OptionTable["file"];

    if (! TMushFileAttribute(ImageFileName).IsReadable()) {
	string ImageFilePath = TMushFileAttribute(ImageFileName).PathName();
	if (ImageFilePath.empty() && (! _RootPath.empty())) {
	    ImageFileName = _RootPath + ImageFileName;
	}
    }

    GtkWidget* Widget = gtk_image_new_from_file(ImageFileName.c_str());

    gtk_box_pack_start(GTK_BOX(_CurrentHBox), Widget, FALSE, FALSE, 3);
}

void TKinokoControlPanelGtk::AddVSpace(map<string, string>& OptionTable)
{
    if (_CurrentMenu) {
#if 0	
	//... old GTK library does not have this function ...//
	GtkWidget* MenuItem = gtk_separator_menu_item_new();
#else
	GtkWidget* MenuItem = gtk_menu_item_new();
#endif

	gtk_menu_append(GTK_MENU(_CurrentMenu), MenuItem);
	gtk_widget_show(MenuItem);
    }
    else {
	AddNewLine(OptionTable);
	AddHSpace(OptionTable);
	AddNewLine(OptionTable);
    }
}

void TKinokoControlPanelGtk::AddHSpace(map<string, string>& OptionTable)
{
    if (_IsInToolBar) {
	gtk_toolbar_append_space(GTK_TOOLBAR(_ToolBar));
    }
    else {
	GtkWidget* HPaddingWidget = gtk_label_new("");
	gtk_box_pack_start(GTK_BOX(_CurrentHBox), HPaddingWidget, TRUE, TRUE, 0);
    }
}

void TKinokoControlPanelGtk::AddNewLine(map<string, string>& OptionTable)
{
    _CurrentHBox = gtk_hbox_new(FALSE, 0);
    gtk_box_pack_start(GTK_BOX(_CurrentVBox), _CurrentHBox, FALSE, FALSE, 0);
}

void TKinokoControlPanelGtk::AddViewlet(const std::string& ViewletTypeName, std::map<std::string, std::string>& OptionTable)
{
    string Name = OptionTable["name"];
    string Width = OptionTable["width"];
    string Height = OptionTable["height"];
    string Padding = OptionTable["padding"];

    string Color = OptionTable["color"]; // color for drawing //
    string Foreground = OptionTable["foreground"]; // color for frame etc. //
    string Background = OptionTable["background"];

    GtkWidget* DrawingAreaWidget = gtk_drawing_area_new();

    TKinokoControlImagePort* ImagePort = new TKinokoControlImagePortGtk(
	DrawingAreaWidget
    );

    TKinokoControlViewlet* Viewlet;
    if (ViewletTypeName == "canvas") {
	Viewlet = new TKinokoControlCanvasViewlet(
	    Name, ImagePort
	);
    }
    else if (ViewletTypeName == "plot") {
	double XMin, XMax, YMin, YMax;
	bool IsRangeSpecified = true;

	if (! (istringstream(OptionTable["xmin"]) >> XMin)) {
	    IsRangeSpecified = false;
	}
	if (! (istringstream(OptionTable["xmax"]) >> XMax)) {
	    IsRangeSpecified = false;
	}
	if (! (istringstream(OptionTable["ymin"]) >> YMin)) {
	    IsRangeSpecified = false;
	}
	if (! (istringstream(OptionTable["ymax"]) >> YMax)) {
	    IsRangeSpecified = false;
	}

	if (IsRangeSpecified) {
	    Viewlet = new TKinokoControlPlotViewlet(
		Name, ImagePort, XMin, XMax, YMin, YMax
	    );
	}
	else {
	    Viewlet = new TKinokoControlPlotViewlet(Name, ImagePort);
	}
    }
    else {
	//...
	cerr << "ERROR: unknown viewlet name: " << ViewletTypeName << endl;
	return;
    }

    int WidthValue = 320;
    int HeightValue = 240;
    if (! Width.empty()) {
	istringstream(Width) >> WidthValue;
    }
    if (! Height.empty()) {
	istringstream(Height) >> HeightValue;
    }

    int PaddingValue = 1;
    if (! Padding.empty()) {
	istringstream(Padding) >> PaddingValue;
    }

    _Control->AddWidget(Viewlet);
    _Control->AddImageWidget(Viewlet);
    
    if (! Color.empty()) {
	Viewlet->SetColor(Color);
    }
    if (! Foreground.empty()) {
	Viewlet->SetForegroundColor(Foreground);
    }
    if (! Background.empty()) {
	Viewlet->SetBackgroundColor(Background);
    }

    gtk_signal_connect(
	GTK_OBJECT(DrawingAreaWidget), "expose_event", 
	GTK_SIGNAL_FUNC(expose_event_cb), NULL
    );

    gtk_drawing_area_size(
	GTK_DRAWING_AREA(DrawingAreaWidget), WidthValue, HeightValue
    );
    gtk_box_pack_start(
	GTK_BOX(_CurrentHBox), DrawingAreaWidget, FALSE, FALSE, PaddingValue
    );
}

TKinokoControlVisualWidget* TKinokoControlPanelGtk::AddVisualWidget(TKinokoControlVisualWidgetCreator* Creator, std::map<std::string, std::string>& OptionTable)
{
    string Name = OptionTable["name"];
    string Width = OptionTable["width"];
    string Height = OptionTable["height"];
    string Padding = OptionTable["padding"];

    GtkWidget* DrawingAreaWidget = gtk_drawing_area_new();

    TKinokoControlImagePort* ImagePort = new TKinokoControlImagePortGtk(
	DrawingAreaWidget
    );

    TKinokoControlVisualWidget* VisualWidget = Creator->CreateInstance(
	Name, OptionTable, ImagePort
    );

    int WidthValue = VisualWidget->DefaultWidth();
    int HeightValue = VisualWidget->DefaultHeight();
    if (! Width.empty()) {
	istringstream(Width) >> WidthValue;
    }
    if (! Height.empty()) {
	istringstream(Height) >> HeightValue;
    }

    int PaddingValue = 1;
    if (! Padding.empty()) {
	istringstream(Padding) >> PaddingValue;
    }

    _Control->AddWidget(VisualWidget);
    _Control->AddImageWidget(VisualWidget);
    
    if (! Name.empty()) {
	_Control->AddInputWidget(VisualWidget);
    }

    gtk_signal_connect(
	GTK_OBJECT(DrawingAreaWidget), "expose_event", 
	GTK_SIGNAL_FUNC(expose_event_cb), NULL
    );
    gtk_signal_connect(
	GTK_OBJECT(DrawingAreaWidget), "button_press_event", 
	GTK_SIGNAL_FUNC(button_press_event_cb), VisualWidget
    );
    gtk_signal_connect(
	GTK_OBJECT(DrawingAreaWidget), "button_release_event", 
	GTK_SIGNAL_FUNC(button_release_event_cb), VisualWidget
    );
    gtk_widget_set_events(
	GTK_WIDGET(DrawingAreaWidget), 
	GDK_EXPOSURE_MASK | GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK
    );

    gtk_drawing_area_size(
	GTK_DRAWING_AREA(DrawingAreaWidget), WidthValue, HeightValue
    );
    gtk_box_pack_start(
	GTK_BOX(_CurrentHBox), DrawingAreaWidget, FALSE, FALSE, PaddingValue
    );

    return VisualWidget;
}

void TKinokoControlPanelGtk::BuildFontNameTable(void)
{
    //... for backward compatibility ...//
    _PredefinedFontNameTable["times"] = "times";
    _PredefinedFontNameTable["times-roman"] = "times";
    _PredefinedFontNameTable["times-italic"] = "times italic";
    _PredefinedFontNameTable["times-bold"] = "times bold";
    _PredefinedFontNameTable["times-bolditalic"] = "times bold italic";
    _PredefinedFontNameTable["helvetica"] = "helvetica";
    _PredefinedFontNameTable["helvetica-oblique"] = "helvetica oblique";
    _PredefinedFontNameTable["helvetica-bold"] = "helvetica bold";
    _PredefinedFontNameTable["helvetica-boldoblique"] = "helvetica bold oblique";
    _PredefinedFontNameTable["courier"] = "courier";
    _PredefinedFontNameTable["courier-oblique"] = "courier oblique";
    _PredefinedFontNameTable["courier-bold"] = "courier bold";
    _PredefinedFontNameTable["courier-boldoblique"] = "courier bold oblique";
    _PredefinedFontNameTable["symbol"] = "symbol";
}

PangoFontDescription* TKinokoControlPanelGtk::LoadFont(string FontName, string FontSize)
{
    if (FontName.empty()) {
        FontName = "helvetica";
    }
    if (FontSize.empty()) {
        FontSize = "12";
    }

    string PangoFontName;
    if (_PredefinedFontNameTable.count(FontName) > 0) {
	PangoFontName = (
            _PredefinedFontNameTable[FontName] + " " + FontSize
	);
    }
    else {
	PangoFontName = FontName;
	if (! isdigit(*(PangoFontName.end()-1))) {
	    PangoFontName += " " + FontSize;
	}
    }
    
    return pango_font_description_from_string(PangoFontName.c_str());
}



static void widget_clicked_cb(GtkWidget* Widget, gpointer Data)
{
    if (g_Control != 0) {
	g_Control->OnClickEvent((TKinokoControlWidget*) Data);
    }
}

static gint button_press_event_cb(GtkWidget* Widget, GdkEventButton* Event, gpointer* Data)
{
    ((TKinokoControlVisualWidget*) Data)->OnButtonPress(Event->button);

    if (g_Control != 0) {
	g_Control->OnClickEvent((TKinokoControlWidget*) Data);
    }

    return FALSE;
}

static gint button_release_event_cb(GtkWidget* Widget, GdkEventButton* Event, gpointer* Data)
{
    ((TKinokoControlVisualWidget*) Data)->OnButtonRelease(Event->button);

    return FALSE;
}

static gboolean widget_focus_event_cb(GtkWidget* Widget, GdkEventFocus* event, gpointer Data)
{
    if (g_Control != 0) {
	if (event->in) {
	    g_Control->OnFocusEvent((TKinokoControlWidget*) Data);
	}
	else {
	    g_Control->OnBlurEvent((TKinokoControlWidget*) Data);
	}
    }

    return FALSE;
}

static void menu_save_as_cb(void)
{
    if (g_Control == 0) {
	return;
    }

    string FileName = TKinokoShellFileSelectDialogGtk().Open();
    if (! FileName.empty()) {
	g_Control->SaveWidgetValues(FileName);
    }
}

static void menu_load_cb(void)
{
    if (g_Control == 0) {
	return;
    }

    string FileName = TKinokoShellFileSelectDialogGtk().Open();
    if (! FileName.empty()) {
	g_Control->LoadWidgetValues(FileName);
    }
}

static void menu_quit_cb(void)
{
    if (g_Control == 0) {
	return;
    }

    g_Control->TryToQuit();
}

static void menu_load_script_cb(void)
{
    if (g_Control == 0) {
	return;
    }

    string FileName = TKinokoShellFileSelectDialogGtk().Open();
    if (FileName.empty()) {
	return;
    }

    ifstream ScriptFile(FileName.c_str());
    if (! ScriptFile) {
	cerr << "ERROR: unable to open file: " + FileName << endl;
	return; //...
    }

    try {
	g_Control->LoadControlScript(ScriptFile);
    }
    catch (TScriptException& e) {
	cerr << "ERROR: " << e << endl;
	return; //...
    }
}

static void menu_about_cb(void)
{
    TKinokoShellAboutDialogGtk AboutDialog;

    AboutDialog.SetTitle("Kinoko Control " KINOKO_SHELL_VERSION);
    AboutDialog.SetCopyright(KINOKO_SHELL_COPYING);
    AboutDialog.AddAuthor(KINOKO_SHELL_AUTHOR);
    AboutDialog.AddComment("Remotely-accessible script-based GUI control panel");

    AboutDialog.Open();
}

static gint delete_event_cb(GtkWidget* Widget, GdkEventAny* Event, gpointer Data)
{
    bool Result = g_Control->TryToQuit();
    return Result ? FALSE : TRUE;
}

static gint expose_event_cb(GtkWidget* Widget, GdkEventExpose* Event)
{
    if (g_Control) {
	g_Control->RedrawImageWidgets();
    }

    return FALSE;
}

static void file_select_cb(GtkWidget* Widget, gpointer Data)
{
    GtkWidget* EntryWidget = GTK_WIDGET(Data);
    string InitialFileName = gtk_entry_get_text(GTK_ENTRY(EntryWidget));

    string SelectedFile = TKinokoShellFileSelectDialogGtk().Open(
	InitialFileName
    );

    if (! SelectedFile.empty()) {
	gtk_entry_set_text(GTK_ENTRY(EntryWidget), SelectedFile.c_str());
    }
}

#if 1
//... for backward compatibility ...//
static void increment_cb(GtkWidget* Widget, gpointer Data)
{
    GtkWidget* EntryWidget = GTK_WIDGET(Data);
    string OriginalValue = gtk_entry_get_text(GTK_ENTRY(EntryWidget));

    long Value = 0;
    if (! (istringstream(OriginalValue) >> Value)) {
	Value = 0;
    }
    ++Value;

    ostringstream NewValueStream;
    NewValueStream << setw(OriginalValue.size()) << setfill('0') << Value;

    gtk_entry_set_text(GTK_ENTRY(EntryWidget), NewValueStream.str().c_str());
}
#endif
