/* kinoko-monitor-gtk.cc */
/* Created by Enomoto Sanshiro on 16 May 2004. */
/* Last updated by Enomoto Sanshiro on 16 May 2004. */


#include <iostream>
#include <iomanip>
#include <string>
#include <gtk/gtk.h>
#include "MushTimer.hh"
#include "MushMisc.hh"
#include "KinokoShellAboutDialogGtk.hh"
#include "KinokoShellConfig.hh"
#include "KinokoMonitor.hh"


using namespace std;


class TKinokoMonitorGtk: public TKinokoMonitor {
  public:
    TKinokoMonitorGtk(int argc, char** argv);
    virtual ~TKinokoMonitorGtk();
    virtual void Start(void);
    virtual void Quit(void);
    virtual void Transact(void);
  protected:
    virtual void CheckStatus(void);
  protected:
    void CreateRootWindow(int Width, int Height);
  protected:
    std::string _RegistryFileName;
  protected:
    GtkWidget *_RootWindow;
    GtkWidget *_ComponentCList;
    GtkWidget *_FileCList;
    GtkWidget *_Statusbar;
};


static void menu_exit_cb(void);
static void menu_about_cb(void);
static gint delete_event_cb(GtkWidget* Widget, GdkEventAny* Event, gpointer Data);
static gint timeout_event_cb(gpointer Data);


TKinokoMonitorGtk::TKinokoMonitorGtk(int argc, char** argv)
{
    _RegistryFileName = argv[1];
    gtk_init(&argc, &argv);
}

TKinokoMonitorGtk::~TKinokoMonitorGtk()
{
}

void TKinokoMonitorGtk::Start(void)
{
    CreateRootWindow(640, 400);
    gtk_timeout_add(1000, timeout_event_cb, this);
    gtk_main();
}

void TKinokoMonitorGtk::Quit(void)
{
    gtk_main_quit();
}

void TKinokoMonitorGtk::Transact(void)
{
    Update(_RegistryFileName);
    CheckStatus();
}

void TKinokoMonitorGtk::CheckStatus(void)
{
    gtk_clist_clear(GTK_CLIST(_ComponentCList));    
    for (unsigned i = 0; i < _ComponentNameList.size(); i++) {
	const string& Name = _ComponentNameList[i];
	const TKinokoMonitorComponentCondition* Condition = _ComponentList[Name];
	string CheckTime = TMushDateTime(Condition->CheckTime()).AsString("%H:%M:%S");
	double HeartBeat = Condition->HeartBeatCount();

	string HeartBeatBar;
	if (TMushDateTime::SecSince(Condition->CheckTime()) > 100) {
	    HeartBeatBar = "DEAD";
	}
	else {
	    while (HeartBeat >= 1) {
		HeartBeatBar += '*';
		HeartBeat /= 2;
	    }
	}

	static const char* CListContentBuffer[4];
	CListContentBuffer[0] = Name.c_str();
	CListContentBuffer[1] = Condition->State().c_str();
	CListContentBuffer[2] = CheckTime.c_str(); 
	CListContentBuffer[3] = HeartBeatBar.c_str();

	//...
	gtk_clist_append(
	    GTK_CLIST(_ComponentCList), (gchar**) CListContentBuffer
	);
    }
    gtk_widget_show_all(_ComponentCList);

    gtk_clist_clear(GTK_CLIST(_FileCList));
    for (unsigned i = 0; i < _FileList.size(); i++) {
	const string& Name = _FileNameList[i];
	const TKinokoMonitorFileCondition* Condition = _FileList[Name];
	string CheckTime = TMushDateTime(Condition->CheckTime()).AsString("%H:%M:%S");
	double GrowingSpeed = Condition->GrowingSpeed();
	string GrowingSpeedBar;
	while (GrowingSpeed >= 1) {
	    GrowingSpeedBar += '*';
	    GrowingSpeed /= 2;
	}

	static const char* CListContentBuffer[4];
	CListContentBuffer[0] = Name.c_str();
	CListContentBuffer[1] = Condition->FileSize().c_str();
	CListContentBuffer[2] = CheckTime.c_str(); 
	CListContentBuffer[3] = GrowingSpeedBar.c_str();

	//...
	gtk_clist_append(
	    GTK_CLIST(_FileCList), (gchar**) CListContentBuffer
	);
    }
    gtk_widget_show_all(_FileCList);
}


/////////////////////


static GtkItemFactoryEntry MenuItemList[] = {
    {"/_File",          NULL,         0,             0, "<Branch>"},
    {"/File/e_Xit",     "<control>X", menu_exit_cb,  0, "<Item>"},
    {"/_Help",          NULL,         0,             0, "<Branch>"},
    {"/Help/_About...", NULL,         menu_about_cb, 0, "<Item>"}
};
static gint NumberOfMenuItems = sizeof(MenuItemList) / sizeof(MenuItemList[0]);


void TKinokoMonitorGtk::CreateRootWindow(int Width, int Height)
{
    // RootWindow //
    _RootWindow = gtk_window_new(GTK_WINDOW_TOPLEVEL);
    gtk_object_set_data(GTK_OBJECT(_RootWindow), "root_window", _RootWindow);
    gtk_window_set_title(GTK_WINDOW(_RootWindow), "kinoko-monitor");
    gtk_window_set_default_size(GTK_WINDOW(_RootWindow), Width, Height);
    
    GtkWidget* RootVBox = gtk_vbox_new(FALSE, 0);
    gtk_widget_ref(RootVBox);
    gtk_object_set_data_full(
	GTK_OBJECT(_RootWindow), "RootVBox", RootVBox,
	(GtkDestroyNotify) gtk_widget_unref
    );
    gtk_widget_show(RootVBox);
    gtk_container_add(GTK_CONTAINER(_RootWindow), RootVBox);
    
    // Menu and Menubar //
    GtkAccelGroup* AccelGroup = gtk_accel_group_new();
    GtkItemFactory* ItemFactory = gtk_item_factory_new(
	GTK_TYPE_MENU_BAR, "<menu>", AccelGroup
    );
    gtk_item_factory_create_items(
	ItemFactory, NumberOfMenuItems, MenuItemList, NULL
    );
    gtk_window_add_accel_group(GTK_WINDOW(_RootWindow), AccelGroup);
    GtkWidget* MenuBar = gtk_item_factory_get_widget(ItemFactory, "<menu>");
    GtkWidget* MenuHandleBox = gtk_handle_box_new();
    gtk_container_add(GTK_CONTAINER(MenuHandleBox), MenuBar);
    gtk_box_pack_start(GTK_BOX(RootVBox), MenuHandleBox, FALSE, FALSE, 0);

    // Component List Paned //
    GtkWidget* ComponentPaned = gtk_vpaned_new();
    gtk_widget_ref(ComponentPaned);
    gtk_object_set_data_full(
	GTK_OBJECT(_RootWindow), "ComponentPaned", ComponentPaned,
	(GtkDestroyNotify) gtk_widget_unref
    );
    gtk_widget_show(ComponentPaned);
    gtk_box_pack_start(GTK_BOX(RootVBox), ComponentPaned, TRUE, TRUE, 0);
    gtk_paned_set_position(GTK_PANED(ComponentPaned), Height/2);
    
    // Component List //
    GtkWidget* ComponentWindow = gtk_scrolled_window_new(NULL, NULL);
    gtk_widget_ref(ComponentWindow);
    gtk_object_set_data_full(
	GTK_OBJECT(_RootWindow), "ComponentWindow", ComponentWindow,
	(GtkDestroyNotify) gtk_widget_unref
    );
    gtk_widget_show(ComponentWindow);
    gtk_paned_pack1(GTK_PANED(ComponentPaned), ComponentWindow, FALSE, TRUE);
    
    static const int NumberOfComponentMonitorFields = 4;
    static const char* ComponentMonitorFieldList[] = { 
	"Component", "State", "Check Time", "Heart Beat"
    };
    _ComponentCList = gtk_clist_new(NumberOfComponentMonitorFields);
    gtk_widget_ref(_ComponentCList);
    gtk_object_set_data_full(
	GTK_OBJECT(_RootWindow), "ComponentCList", _ComponentCList,
	(GtkDestroyNotify) gtk_widget_unref
    );
    gtk_widget_show(_ComponentCList);
    gtk_container_add(GTK_CONTAINER(ComponentWindow), _ComponentCList);
    gtk_clist_column_titles_show(GTK_CLIST(_ComponentCList));
    
    for (int i = 0; i < NumberOfComponentMonitorFields; i++) {
	const char* FieldName = ComponentMonitorFieldList[i];
	GtkWidget* FieldNameLabel = gtk_label_new(FieldName);
	gtk_widget_ref(FieldNameLabel);
	gtk_object_set_data_full(
	    GTK_OBJECT(_RootWindow), FieldName, FieldNameLabel,
	    (GtkDestroyNotify) gtk_widget_unref
	);
	gtk_widget_show(FieldNameLabel);
	gtk_clist_set_column_widget(
	    GTK_CLIST(_ComponentCList), i, FieldNameLabel
	);
	gtk_clist_set_column_width(GTK_CLIST(_ComponentCList), i, 120);
    }
    
    // File List Paned //
    GtkWidget* FilePaned = gtk_vpaned_new();
    gtk_widget_ref(FilePaned);
    gtk_object_set_data_full(
	GTK_OBJECT(_RootWindow), "FilePaned", FilePaned,
	(GtkDestroyNotify) gtk_widget_unref
    );
    gtk_widget_show(FilePaned);
    gtk_box_pack_start(GTK_BOX(RootVBox), FilePaned, TRUE, TRUE, 0);
    //gtk_paned_set_position(GTK_PANED(FilePaned), Height/2);
    
    // File Box //
    GtkWidget* FileWindow = gtk_scrolled_window_new(NULL, NULL);
    gtk_widget_ref(FileWindow);
    gtk_object_set_data_full(
	GTK_OBJECT(_RootWindow), "FileWindow", FileWindow,
	(GtkDestroyNotify) gtk_widget_unref
    );
    gtk_widget_show(FileWindow);
    //gtk_paned_pack2(GTK_PANED(FilePaned), FileWindow, TRUE, TRUE);
    gtk_paned_pack1(GTK_PANED(FilePaned), FileWindow, FALSE, TRUE);

    static const int NumberOfFileMonitorFields = 4;
    static const char* FileMonitorFieldList[] = { 
	"File", "File Size", "Check Time", "Growing Speed"
    };
    _FileCList = gtk_clist_new(NumberOfFileMonitorFields);
    gtk_widget_ref(_FileCList);
    gtk_object_set_data_full(
	GTK_OBJECT(_RootWindow), "FileCList", _FileCList,
	(GtkDestroyNotify) gtk_widget_unref
    );
    gtk_widget_show(_FileCList);
    gtk_container_add(GTK_CONTAINER(FileWindow), _FileCList);
    gtk_clist_column_titles_show(GTK_CLIST(_FileCList));
    
    for (int i = 0; i < NumberOfFileMonitorFields; i++) {
	const char* FieldName = FileMonitorFieldList[i];
	GtkWidget* FieldNameLabel = gtk_label_new(FieldName);
	gtk_widget_ref(FieldNameLabel);
	gtk_object_set_data_full(
	    GTK_OBJECT(_RootWindow), FieldName, FieldNameLabel,
	    (GtkDestroyNotify) gtk_widget_unref
	);
	gtk_widget_show(FieldNameLabel);
	gtk_clist_set_column_widget(
	    GTK_CLIST(_FileCList), i, FieldNameLabel
	);
	gtk_clist_set_column_width(GTK_CLIST(_FileCList), i, 120);
    }

    // Status Bar //
    _Statusbar = gtk_statusbar_new();
    gtk_widget_ref(_Statusbar);
    gtk_object_set_data_full(
	GTK_OBJECT(_RootWindow), "StatusBar", _Statusbar,
	(GtkDestroyNotify) gtk_widget_unref
    );
    gtk_widget_show(_Statusbar);
    gtk_box_pack_start(GTK_BOX(RootVBox), _Statusbar, FALSE, FALSE, 0);
    
    // Signal Connection //
    gtk_signal_connect(
	GTK_OBJECT(_RootWindow), "delete_event", 
	GTK_SIGNAL_FUNC(delete_event_cb), NULL
    );

    gtk_widget_show_all(_RootWindow);
}

static gint timeout_event_cb(gpointer Data)
{
    TKinokoMonitorGtk* Monitor = (TKinokoMonitorGtk*) Data;
    Monitor->Transact();
    return 1;
}

static void menu_exit_cb(void)
{
    gtk_main_quit();
}

static void menu_about_cb(void)
{
    TKinokoShellAboutDialogGtk AboutDialog;

    AboutDialog.SetTitle("Kinoko Monitor " KINOKO_SHELL_VERSION);
    AboutDialog.SetCopyright(KINOKO_SHELL_COPYING);
    AboutDialog.AddAuthor(KINOKO_SHELL_AUTHOR);
    AboutDialog.AddComment("Kinoko Process/File/Status monitor");

    AboutDialog.Open();
}

static gint delete_event_cb(GtkWidget* Widget, GdkEventAny* Event, gpointer Data)
{
    gtk_main_quit();
    return TRUE;
}


/////////////////////


int main(int argc, char** argv)
{
    if (argc < 2) {
	cerr << "Usage: " << argv[0] << " file_name" << endl;
	return -1;
    }

    TKinokoMonitorGtk(argc, argv).Start();

    return 0;
}
